'use strict';

const _ = require('lodash');
const assert = require('helpers/assert');
const config = require('yandex-config');

const { getPrevMonthInterval } = require('helpers/dateHelper');

const AttemptModel = require('models/attempt');
const BlackboxModel = require('models/blackbox');
const Certificate = require('models/certificate');
const Notifier = require('models/notifier');
const ProctorEdu = require('models/proctoring/proctorEdu');
const ProctoringResponse = require('models/proctoringResponse');
const ProctoringVideos = require('models/proctoringVideos');
const Report = require('models/report');
const S3 = require('models/s3');
const Tracker = require('models/tracker');
const UserIdentification = require('models/userIdentification');
const WebUser = require('models/user/webUser');
const YT = require('models/yt');

// eslint-disable-next-line complexity, max-statements
function *getProcessedTrialsData(trialsToProcess, transaction) {
    const trialsToProcessIds = trialsToProcess.map(trial => trial.id);
    const videosByTrial = yield ProctoringVideos.getVideosByTrial(
        trialsToProcessIds,
        ['webcam'],
        transaction
    );

    const rowsToUpload = [];
    const trialIdsWithoutData = [];
    const trialsWithHighMetrics = [];

    for (const trial of trialsToProcess) {
        const { id: trialId, openId, userId, isRevision } = trial;
        const sessionData = yield ProctorEdu.getSessionData(openId);

        const isMetricsHigh = ProctorEdu.isMetricsHigh(sessionData);

        if (!isRevision && isMetricsHigh) {
            trialsWithHighMetrics.push(trial);

            continue;
        }

        const videos = _.defaultTo(videosByTrial[trialId], []);
        const videosMarkup = ProctorEdu.getVideosMarkup(sessionData, videos, isRevision);

        if (!sessionData || _.isEmpty(videosMarkup)) {
            trialIdsWithoutData.push(trialId);

            yield Tracker.createPendingTicket(trialId);

            continue;
        }

        const userPhoto = yield UserIdentification.getLastFace(userId, transaction);

        rowsToUpload.push({
            trialId,
            userPhoto: S3.getPathToProxyS3('public', userPhoto),
            videos: videosMarkup,
            isRevision
        });

        if (rowsToUpload.length === config.yt.tableCapacity) {
            break;
        }
    }

    return { rowsToUpload, trialIdsWithoutData, trialsWithHighMetrics };
}

function *failTrialsWithHighMetrics(data, transaction) {
    const uids = data.map(trial => _.get(trial, 'userUid'));

    const blackbox = new BlackboxModel(config.blackbox);
    const emails = yield blackbox.getEmails(uids);

    for (const trial of data) {
        const dataForResponse = [{
            source: 'metrics',
            verdict: 'failed'
        }];

        yield ProctoringResponse.create(trial.id, dataForResponse, transaction);

        yield Notifier.sendLetterToUser(emails[trial.userUid], 'failed');
    }
}

module.exports = {
    uploadTrialsToYT: function *uploadTrialsToYT() {
        const { transaction } = this.state;

        const [pendingTrials, trialsToRevision] = yield [
            AttemptModel.getNewPendingTrials(transaction),
            AttemptModel.getNewTrialsForRevision(transaction)
        ];

        const trialsToProcess = pendingTrials.concat(trialsToRevision);

        const {
            trialIdsWithoutData,
            trialsWithHighMetrics,
            rowsToUpload
        } = yield getProcessedTrialsData(trialsToProcess, transaction);

        yield ProctoringResponse.setSentToTolokaByTrialIds(trialIdsWithoutData, transaction);

        if (trialsWithHighMetrics.length > 0) {
            yield failTrialsWithHighMetrics(trialsWithHighMetrics, transaction);
        }

        const trialIdsToUpload = _.map(rowsToUpload, 'trialId');
        const trialsToUpload = trialsToProcess.filter(trial =>
            _.includes(trialIdsToUpload, trial.id)
        );

        const isExpiredTrialExist = AttemptModel.isExpiredTrialExist(trialsToUpload);

        if (isExpiredTrialExist || rowsToUpload.length === config.yt.tableCapacity) {
            yield YT.upload(rowsToUpload);
            yield ProctoringResponse.setSentToTolokaByTrialIds(trialIdsToUpload, transaction);
        }

        this.status = 204;
    },

    // eslint-disable-next-line complexity
    loadTolokaResults: function *loadTolokaResults() {
        const { transaction } = this.state;
        const tables = yield YT.loadResults();
        const violationsByTrial = YT.aggregateResults(tables);
        const trialIds = _.keys(violationsByTrial);

        for (const trialId of trialIds) {
            const { isViolationsExist, isRevision, intervals } = violationsByTrial[trialId];
            const verdict = isViolationsExist ? 'failed' : 'correct';

            const proctoringData = [{
                source: isRevision ? 'toloka-revision' : 'toloka',
                verdict,
                additionalInfo: intervals
            }];

            yield ProctoringResponse.create(parseInt(trialId, 10), proctoringData, transaction);

            if (!isViolationsExist) {
                yield Certificate.create({ id: trialId }, transaction);
            }

            if (isRevision) {
                const { email } = yield WebUser.findUserByTrialId({
                    trialId,
                    attributes: ['email']
                });

                yield Notifier.sendLetterToUser(email, verdict);
            }
        }

        yield YT.moveToArchive(tables);

        this.status = 204;
    },

    *uploadToStat() {
        const { from, to } = getPrevMonthInterval(new Date());

        const data = yield Report.getStatReportsData(from, to);

        yield YT.uploadReports(data);

        this.status = 204;
    },

    *cleanExpiredTables() {
        const allTables = yield YT.getTablesByDirs(config.yt.clean.dirPaths);
        const expiredTables = YT.getExpiredTables(allTables);

        yield YT.removeTables(expiredTables);

        this.status = 204;
    },

    *uploadComdepReport() {
        const chunks = yield Report.getComdepReportData();
        const { dirPath, tableName, schema } = config.comdepReport;

        try {
            yield YT.createTable({
                mapNode: dirPath,
                tableName,
                schema,
                force: true
            });

            for (const chunk of chunks) {
                const uploadChunkData = { [tableName]: chunk };

                yield YT.uploadReports(uploadChunkData, dirPath);
            }
        } catch (err) {
            yield Notifier.failedUpdatingReport(err.message);

            assert(false, 500, 'Upload comdep report was failed', 'URF', { details: err.message });
        }

        this.status = 204;
    }
};
