'use strict';

const bodyParser = require('body-parser');
const config = require('cfg');
const cookieParser = require('cookie-parser');
const express = require('express');
const expressBlackbox = require('express-blackbox');
const expressDomainAccess = require('express-domain-access');
const expressRenewBbSession = require('express-renew-bb-session');
const expressRequestId = require('express-request-id');
const expressSecretkey = require('express-secretkey');
const expressTld = require('express-tld');
const expressTvm = require('@yandex-int/express-tvm').default;
const expressXff = require('express-x-forwarded-for-fix');
const expressYaFrameguard = require('express-ya-frameguard');
const expressYandexCsp = require('express-yandex-csp');
const expressYandexUid = require('@yandex-int/express-yandexuid');
const helmet = require('helmet');
const morgan = require('morgan');

const bunker = require('./middleware/bunker');
const expressBundleResponse = require('./middleware/express-bundle-response');
const geobaseMock = require('./middleware/geobase-mock');
const language = require('./middleware/language');
const mainMenu = require('./middleware/main-menu');
const referrerPolicy = require('./middleware/referrer-policy');
const uatraits = require('./middleware/uatraits');

morgan.token('requestId', req => req.id);

const app = express();

require('./lib/debug-http')();

app.set('env', config.environment);
app.disable('x-powered-by');
app.disable('etag');
app.enable('trust proxy');

app.use(referrerPolicy());

app.use(`${config.router.namespace}/static`, express.static(config.static.dir, config.static.options));

app.use(expressRequestId());
app.use(morgan(':status :requestId :method :url ":referrer" ":user-agent" :res[content-length]'));

app.use(bodyParser.json(config.bodyParser));
app.use(cookieParser());
app.use(expressXff());
app.use(expressYandexCsp(config.csp));

app.use(helmet.noSniff());
app.use(helmet.xssFilter({ setOnOldIE: true }));

app.use(config.router.namespace, express.static(config.www.dir, config.www.options));

app.use(expressTld());
app.use(language(config.tldToLang));
app.use(expressDomainAccess(config.app.domains));
app.use(geobaseMock());
app.use(expressYaFrameguard());
app.use(uatraits());
app.use(expressYandexUid());
app.use(expressTvm(config.tvm.options));
app.use(expressBlackbox(config.blackbox));
app.use(expressRenewBbSession());
app.use(expressSecretkey());
app.use(expressBundleResponse(config.view));
app.use(bunker());

app.use(`${config.router.namespace}/favicon.ico`, function (req, res) {
    const faviconPath = config.favicon[req.tld] || config.favicon.default;

    res.redirect(`${config.router.namespace}${faviconPath}`);
});

// Добавляем основные параметры
app.use(require('./middleware/express-base')());
app.use(mainMenu());

require('./router')(app);

module.exports = app;
