'use strict';

const url = require('url');
const _ = require('lodash');
const config = require('cfg');

const buildPath = require('../lib/build-path');
const { getBunkerNode } = require('../lib/helper');

const BaseController = require('./base');
const AttemptResultModel = require('../model/attempt-result');
const logger = require('../logger');

class AttemptResult extends BaseController {
    get pageName() {
        return 'attempt-result';
    }

    resultPage() {
        var model = new AttemptResultModel(this._req, {});

        model
            .fetch()
            .bind(this)
            .then(this._toJSON)
            .then(this._extendData)
            .then(this._addSeoInfo)
            .then(this._addShareUrl)
            .then(this._addRulesUrl)
            .then(this._addFormId)
            .then(data => {
                return this._addQueryParams(data, ['retlink']);
            })
            .then(this._addExamInfo)
            .then(this._render)
            .catch(this._renderError);
    }

    _addShareUrl(data) {
        const content = data.content;

        if (!_.get(content, 'certId')) {
            return data;
        }

        const urlData = {
            protocol: 'https',
            host: data.host
        };
        if (content.isCertification) {
            _.assign(urlData, {
                pathname: buildPath('certificates'),
                query: _.pick(content, ['certId', 'lastname'])
            });
            content.certUrl = url.format(urlData);
        } else if (content.isAchievement) {
            _.assign(urlData, {
                pathname: buildPath('certificates', content.hashedUserId),
                query: {
                    id: content.certId
                }
            });
            content.achievementUrl = url.format(urlData);
        }

        return data;
    }

    _addRulesUrl(data) {
        const isProctoring = _.get(data, 'content.isProctoring');
        const rulesPath = isProctoring ? 'pro-rules#proctoring' : 'pro-rules';
        const rulesUrl = buildPath(rulesPath);

        return _.merge({ content: { rulesUrl }}, data);
    }

    _addFormId(data) {
        const { isRevisionRequested, lastSource } = _.get(data, 'content.total', {});
        const isCritMetrics = lastSource === 'crit-metrics';

        const formName = (isRevisionRequested || isCritMetrics) ? 'support' : 'toloka';
        const formId = _.get(config, ['proctoringForms', this.pageName, formName]);

        return _.set(data, 'formId', formId);
    }

    _isMatchHostPattern(host, patterns) {
        if (!patterns) {
            return false;
        }

        return patterns.some(pattern => {
            const hostEls = host.split('.');
            const patternEls = pattern.split('.');

            if (hostEls.length !== patternEls.length) {
                return false;
            }

            for (let i = 0; i < patternEls.length; i++) {
                if (patternEls[i] !== '*' && patternEls[i] !== hostEls[i]) {
                    return false;
                }
            }

            return true;
        });
    }

    _addExamInfo(data) {
        const examSlug = this._req.params.examSlug;
        data.content.isHiddenExam = config.hiddenExams.includes(examSlug);

        const examBunkerData = getBunkerNode(this._req, ['exams', examSlug]);
        const addInfo = _.get(examBunkerData, 'additionalInfo');

        if (data.retlink && addInfo) {
            try {
                const urlParsed = new URL(data.retlink);
                if (this._isMatchHostPattern(urlParsed.host, addInfo.allowedRedirectHosts)) {
                    addInfo.link = data.retlink;
                }
            } catch (e) {
                logger.error('Incorrect retlink: ' + data.retlink);
            }
        }

        data.content.additionalInfo = addInfo;
        return data;
    }
}
module.exports = AttemptResult;
