'use strict';

const Promise = require('bluebird');
const _ = require('lodash');
const config = require('cfg');

const BaseController = require('./base');
const AttemptModel = require('../model/attempt');
const ExamModel = require('../model/exam');
const ExamAccessModel = require('../model/exam-access');

const buildPath = require('../lib/build-path');
const log = require('../logger');
const {
    getProctoringFooter,
    getProctoringSettings,
    createUserJWT,
    isValidBrowser,
    buildQueryStr
} = require('../lib/helper');

class Attempt extends BaseController {
    get pageName() {
        return 'attempt';
    }

    get resultPageUrl() {
        return buildPath('exam', this._req.params.examSlug, this._req.params.attemptId, 'result') +
            buildQueryStr({ retlink: this._req.query.retlink });
    }

    attemptPage() {
        Promise
            .all([
                new AttemptModel(this._req).fetch(),
                new AttemptModel(this._req)
                    .getProctoringMetrics()
                    .catch(err => {
                        log.error(err);

                        return {};
                    }),
                new ExamAccessModel(this._req).fetch()
            ])
            .bind(this)
            .then(responses => {
                const [data, proctoringMetrics, examData] = responses.map(this._toJSON);

                return _.assign({ proctoringMetrics }, data, _.pick(examData, ['openId', 'attemptUrl']));
            })
            .then(data => {
                if (data.exam && data.exam.isProctoring && data.openId !== this._req.query.openId) {
                    this._res.redirect(data.attemptUrl);
                    throw new Promise.CancellationError('Incorrect or no openId for started attempt, redirect user to attempt page');
                }

                return data;
            })
            .then(data => {
                if (!data.hasOwnProperty('questionId')) {
                    this._redirectToResult();
                    throw new Promise.CancellationError('There are no `questionId` in data, redirect user to result page');
                }

                return data;
            })
            .then(this._addBunkerContent)
            .then(this._extendData)
            .then(this._addCommonInfo)
            .then(this._addSeoInfo)
            .then(this._addFormId)
            .then(this._addProctoringIframeUrl)
            .then(data => {
                return this._addQueryParams(data, ['openId', 'retlink']);
            })
            .then(this._addUserJWT)
            .then(this._renderPage)
            .catch(Promise.CancellationError, error => {
                log.info(error.message);
            })
            .catch(err => this._detectError(err, '403_TFN') ? this._verifyNext({}) : this._renderError(err));
    }

    create() {
        const isMobile = _.get(this._req, 'base.uatraits.isMobile');
        const exam = new ExamModel(this._req);
        const attempt = new AttemptModel(this._req);

        exam
            .fetch()
            .bind(this)
            .then(this._toJSON)
            .then(data => {
                if (isMobile && data.isProctoring) {
                    return this._res.redirect(buildPath('exam', this._req.params.examSlug));
                }

                return attempt.create();
            })
            .bind(this)
            .then(this._toJSON)
            .then(this._onCreateSuccess)
            .catch(err => {
                if (this._detectError(err, '403_AAS')) {
                    return this._onCreateSuccess({
                        examSlug: attempt.examSlug,
                        id: _.get(err, 'response.body.attemptId')
                    });
                }

                if (this._detectError(err, '400_OIA')) {
                    return this._res.redirect(buildPath('exam', attempt.examSlug, 'go'));
                }

                if (this._detectError(err, ['403_UIB', '403_ANA', '403_LAC', '403_UIS'])) {
                    return this._res.redirect(buildPath('exam', attempt.examSlug));
                }

                this._renderError(err);
            });
    }

    nextQuestion() {
        const model = new AttemptModel(this._req);

        model
            .fetch()
            .bind(this)
            .then(this._toJSON)
            .then(this._verifyNext)
            .catch(err => this._detectError(err, '403_TFN') ? this._verifyNext({}) : this._sendJsonError(err));
    }

    finish() {
        const attempt = new AttemptModel(this._req);
        const { isCritMetrics, requestNullify } = this._req.body;

        attempt
            .finish(isCritMetrics, requestNullify)
            .bind(this)
            .then(response => {
                if (isCritMetrics) {
                    return this._sendJson(response);
                }

                return this._sendRedirectUrl();
            })
            .catch(err => {
                if (this._detectError(err, '403_TFN')) {
                    return this._verifyNext({});
                }

                if (this._detectError(err, '403_AAF')) {
                    return this._res.redirect(buildPath('exam', attempt.examSlug, attempt.attemptId, 'result'));
                }

                this._sendJsonError(err);
            });
    }

    answer() {
        const model = new AttemptModel(this._req);

        model
            .answer()
            .bind(this)
            .then(this._sendOk)
            .catch(err => this._detectError(err, '403_TFN') ? this._verifyNext({}) : this._sendJsonError(err));
    }

    saveProctoringMetrics() {
        const model = new AttemptModel(this._req);

        model
            .saveProctoringMetrics()
            .bind(this)
            .then(this._sendOk)
            .catch(err => this._sendJsonError(err));
    }

    _addBunkerContent(data) {
        const isProctoring = _.get(data, 'exam.isProctoring', false);

        return _.assign({
            proctoringFooter: getProctoringFooter(this._req, isProctoring),
            proctoringSettings: getProctoringSettings(this._req, isProctoring)
        }, data);
    }

    _addCommonInfo(data) {
        data.serverTime = new Date();
        return data;
    }

    _onCreateSuccess(data) {
        const params = _.pick(this._req.query, ['openId', 'retlink']);
        const query = buildQueryStr(params);
        const redirectUrl = `${buildPath('exam', data.examSlug, data.id)}${query}`;

        this._res.redirect(redirectUrl);
    }

    _redirectToResult() {
        this._res.redirect(this.resultPageUrl);
    }

    _verifyNext(data) {
        if (data.hasOwnProperty('questionId')) {
            return this._sendJson(data);
        }

        const { sessionStopped } = this._req.query;

        if (!sessionStopped) {
            return this._redirectToResult();
        }

        if (sessionStopped === '1') {
            return this.finish();
        }

        this._sendJson({ stopSession: true });
    }

    _sendRedirectUrl() {
        this._sendJson({ redirectUrl: this.resultPageUrl });
    }

    _addUserJWT(data) {
        if (!_.get(data, 'content.exam.isProctoring')) {
            return data;
        }

        const { userProctoringToken } = createUserJWT({
            timeLimit: data.content.exam.timeLimit,
            username: _.get(data, 'userInfo.login', 'Some.user'),
            examTitle: data.content.exam.title,
            openId: data.openId
        });

        data.userProctoringToken = userProctoringToken;

        return data;
    }

    _renderPage(data) {
        const isProctoring = _.get(data, 'content.exam.isProctoring');
        const isBrowserSupportProctoring = isValidBrowser(data.uatraits, config.proctoringSupportBrowsers);

        if (isProctoring && !isBrowserSupportProctoring) {
            return this._res.redirect(buildPath('exam', this._req.params.examSlug));
        }

        this._render(data);
    }
}
module.exports = Attempt;
