'use strict';
const _ = require('lodash');
const url = require('url');
const config = require('cfg');
const log = require('../logger');
const buildPath = require('../lib/build-path');
const helper = require('../lib/helper');

class Base {
    /**
     * Возвращает словарь обработчиков ошибок по их кодам.
     * Можно переопределить в дочернем контроллере и указать там другие обработчики.
     * @returns {Object}
     */
    get errorHandlers() {
        return {
            401: '_onUserUnauthorized',
            500: '_onServerError'
        };
    }

    constructor(req, res, next) {
        this._req = req;
        this._res = res;
        this._next = next;
    }

    _render(data) {
        const specificExamLangs = config.specificExamLangs;
        const examSlug = _.get(data, 'content.exam.slug', '');
        const examLang = specificExamLangs[examSlug];

        if (examLang) {
            this._req.language = examLang;
        }

        this._res.bundle.render(this.pageName, data);
    }

    _extendData(data) {
        return _.assign(this._req.base, { content: data, page: this.pageName });
    }

    /**
     * Добавляет в ответ поле `seo`, на основе которого подставляются title, description, keywords
     * Подробнее в common.blocks/seo
     * @param {Object} data
     * @returns {*}
     * @private
     */
    _addSeoInfo(data) {
        const seoPath = this.seoPath || this.pageName;
        const bunkerSeoInfo = helper.getBunkerNode(this._req, ['seo', seoPath], {});
        const seoInfo = _.mapValues(bunkerSeoInfo, value => helper.replaceTemplates(value, data));

        return _.set(data, 'seo', seoInfo);
    }

    /**
     * Добавляет этаж подписок
     * @param {Object} data
     * @returns {Object}
     * @private
     */
    _addShareLevel(data) {
        const settings = helper.getBunkerNode(this._req, 'settings');
        const shareLevel = _.get(settings, 'share-level');

        return _.set(data, 'shareLevel', shareLevel);
    }

    _addFormId(data) {
        const formId = _.get(config, ['proctoringForms', this.pageName]);

        return _.set(data, 'formId', formId);
    }

    /**
     * Добавляет доступность прокторинга в браузере
     * @param {Object} data
     * @returns {Object}
     */
    _addIsBrowserSupportProctoring(data) {
        const isBrowserSupportProctoring = helper.isValidBrowser(data.uatraits, config.proctoringSupportBrowsers);

        return _.set(data, 'isBrowserSupportProctoring', isBrowserSupportProctoring);
    }

    /**
     * Добавляет урлы для вставки iframe прокторинга
     * @param data
     * @returns {Object}
     * @private
     */
    _addProctoringIframeUrl(data) {
        data.proctoringIframeUrl = url.format(config.proctoringIframeUrl);
        data.proctoringIframeHost = url.format({
            protocol: config.proctoringIframeUrl.protocol,
            host: config.proctoringIframeUrl.host
        });

        return data;
    }

    /**
     * Добавляет к data параметры из query по списку
     * @param {Object} data
     * @param {String[]} paths
     * @returns {Object}
     */
    _addQueryParams(data, paths) {
        for (const p of paths) {
            const value = this._req.query[p];
            if (value) {
                data[p] = value;
            }
        }

        return data;
    }

    _renderError(err) {
        const handler = this.errorHandlers[err.statusCode];

        if (handler) {
            this[handler]();
        } else {
            this._res.sendStatus(404);

            log.warn(err, 'Render error');
        }
    }

    _sendOk() {
        this._res.sendStatus(200);
    }

    _sendJson(data) {
        data = data || {};

        this._res.json(data);
    }

    _sendJsonError(err) {
        const errorJson = _.get(err, 'response.body', {});

        this._res
            .status(400)
            .json(errorJson);
    }

    _detectError(err, code) {
        const codes = Array.isArray(code) ? code : [code];
        const internalCode = _.get(err, 'response.body.internalCode');

        return codes.includes(internalCode);
    }

    /**
     * Преобразование всех моделей к JSON формату
     * @param {Object} data
     * @returns {*}
     * @private
     */
    _toJSON(data) {
        if (data.toJSON) {
            return data.toJSON();
        }

        Object
            .keys(data)
            .forEach(key => {
                data[key] = data[key].toJSON();
            });

        return data;
    }

    /**
     * При 401 ошибке отправляем пользователя на страницу auth, передавая текущий URL в retpath
     * @private
     */
    _onUserUnauthorized() {
        const redirectUrl = url.format({
            pathname: buildPath('auth'),
            query: {
                retpath: this._req.base.retpath
            }
        });

        this._res.redirect(redirectUrl);
    }

    _onServerError() {
        return this._res.sendStatus(500);
    }
}

module.exports = Base;
