'use strict';

const _ = require('lodash');
const Promise = require('bluebird');

const config = require('cfg');

const BaseController = require('./base');
const ExamModel = require('../model/exam');
const ExamAccessModel = require('../model/exam-access');
const getBunkerNode = require('../lib/helper').getBunkerNode;

class Exam extends BaseController {
    get pageName() {
        return 'exam';
    }

    get examSlug() {
        return _.get(this._req, 'params.examSlug');
    }

    examPage() {
        const exam = new ExamModel(this._req);
        const examAccess = new ExamAccessModel(this._req);

        Promise.props({
            exam: exam.fetch(),
            examAccess: examAccess.fetch()
        })
            .bind(this)
            .then(super._toJSON)
            .then(data => {
                return this._addQueryParams(data, ['retlink']);
            })
            .then(this._addBunkerContent)
            .then(this._addMetaData)
            .then(super._extendData)
            .then(this._addSeoInfo)
            .then(this._addShareLevel)
            .then(this._addIsBrowserSupportProctoring)
            .then(super._render)
            .catch(super._renderError);
    }

    _addBunkerContent(data) {
        const examBunkerData = getBunkerNode(this._req, ['exams', this.examSlug]);
        data.terms = _.get(examBunkerData, 'terms');

        data.maintenance = getBunkerNode(this._req, ['settings', 'maintenance']);

        const maintenanceForExam = _.find(data.maintenance.exams, { slug: this.examSlug });
        data.maintenance.isFrozen = _.get(maintenanceForExam, 'isFrozen');

        return data;
    }

    _addMetaData(data) {
        const isNoIndex = config.hiddenExams.includes(this.examSlug);

        data.meta = { ...data.meta, noindex: isNoIndex };

        return data;
    }
}
module.exports = Exam;
