'use strict';

const _ = require('lodash');
const Promise = require('bluebird');

const BaseController = require('./base');
const buildPath = require('../lib/build-path');
const ExamModel = require('../model/exam');
const ExamAccessModel = require('../model/exam-access');
const {
    getBunkerNode,
    getProctoringFooter,
    createUserJWT,
    getBrowserSpecificData
} = require('../lib/helper');
const IdentificationModel = require('../model/identification');

class IdentificationSteps extends BaseController {
    get pageName() {
        return 'identification';
    }

    go() {
        const exam = new ExamModel(this._req);
        const examAccess = new ExamAccessModel(this._req);

        Promise.props({
            exam: exam.fetch(),
            examAccess: examAccess.fetch()
        })
            .bind(this)
            .then(super._toJSON)
            .then(this._addBunkerContent)
            .then(this._extendData)
            .then(this._addSeoInfo)
            .then(this._addFormId)
            .then(this._addUserJWT)
            .then(this._addProctoringIframeUrl)
            .then(this._renderPage)
            .catch(this._renderError);
    }

    identificate() {
        const identification = new IdentificationModel(this._req);

        identification
            .fetch()
            .bind(this)
            .then(this._sendJson)
            .catch(this._sendJsonError);
    }

    _addBunkerContent(data) {
        const isProctoring = _.get(data, 'exam.isProctoring', false);
        const identification = getBunkerNode(this._req, ['identification']);

        if (!identification) {
            throw new Error('Conditions not found');
        }

        return _.assign({
            identification,
            proctoringFooter: getProctoringFooter(this._req, isProctoring),
            screencastUrl: getBrowserSpecificData({
                uatraits: this._req.base.uatraits,
                data: _.get(identification, 'screencast.screencasts')
            }),
            identificationUrl: buildPath.ajax('exam', this._req.params.examSlug, 'identificate')
        }, data);
    }

    _renderPage(data) {
        const isProctoring = _.get(data, 'content.exam.isProctoring');
        const isAvailable = _.get(data, 'content.examAccess.state') === 'enabled';
        const isMobile = _.get(data, 'uatraits.isMobile');
        const isAuthorized = Boolean(data.userInfo.uid);

        if (isProctoring && isAvailable && isAuthorized && !isMobile) {
            return this._render(data);
        }

        this._res.redirect(buildPath('exam', this._req.params.examSlug));
    }

    _addUserJWT(data) {
        const { openId, userProctoringToken } = createUserJWT({
            timeLimit: data.content.exam.timeLimit,
            username: _.get(data, 'userInfo.login', 'Some.user'),
            examTitle: data.content.exam.title
        });

        data.userProctoringToken = userProctoringToken;
        data.openId = openId;

        return data;
    }
}

module.exports = IdentificationSteps;
