'use strict';

const urlParse = require('url').parse;
const humanize = require('humanize-number');
const chalk = require('chalk');
const { EventEmitter } = require('events');
const monkeypatch = require('monkeypatch');
const http = require('http');
const log = require('../logger');
const { excludedServices } = require('cfg').logger;
const _ = require('lodash');

const colorCodes = {
    5: 'red',
    4: 'yellow',
    3: 'cyan',
    2: 'green',
    1: 'green'
};

function time(start) {
    let delta = new Date() - start;

    delta = delta < 10000 ? delta + 'ms' : Math.round(delta / 1000) + 's';

    return humanize(delta);
}

function defaultHandler(request, options, cb) {
    options = typeof options === 'string' ? urlParse(options) : options;

    const url = options.href || (options.protocol || 'http://') + options.host + options.path;
    const method = (options.method || 'GET').toUpperCase();
    const signature = method + ' ' + url;
    const start = new Date();
    const wasHandled = typeof cb === 'function';
    const isExcludedUrl = _.some(excludedServices, service => url.startsWith(`http://${service}`));

    if (!isExcludedUrl) {
        setImmediate(log.info.bind(log, chalk.gray('      → ' + signature)));
    }

    return request(options, cb)
        .on('response', function (response) {
            // Workaround for res._dump in Node.JS http client
            // https://github.com/nodejs/node/blob/20285ad17755187ece16b8a5effeaa87f5407da2/lib/_http_client.js#L421-L427
            if (!wasHandled && EventEmitter.listenerCount(response.req, 'response') === 0) {
                response.resume();
            }

            const status = response.statusCode;
            const s = status / 100 | 0;

            if (!isExcludedUrl) {
                log.info(`${chalk[colorCodes[s]](status)} ← ${signature} ${chalk.gray(time(start))}`);
            }
        })
        .on('error', function (err) {
            log.error(`${chalk.red('xxx')} ← ${signature} ${chalk.red(err.message)}`);
        });
}

module.exports = function debugHttp(fn) {
    fn = fn || defaultHandler;

    if (typeof fn !== 'function') {
        throw new TypeError('Expected a function as request handler');
    }

    monkeypatch(http, 'request', fn);
};
