'use strict';

const config = require('cfg');
const _ = require('lodash');
const buildPath = require('../lib/build-path');
const uuid = require('uuid/v4');
const jwt = require('jsonwebtoken');

/**
 * Возвращает нужную ноду, основываясь на переданном tld.
 * Словарь соответствия tld и языковой папки лежит в config.bunkerLang.
 * @param {Object} req
 * @param {String|Array} [path]
 * @param {*} [defaultValue]
 * @returns {*}
 */
function getBunkerNode(req, path, defaultValue) {
    var bunker = req.bunker;

    if (_.isEmpty(bunker)) {
        return defaultValue;
    }

    var lang = config.bunkerLang[req.tld] || config.bunkerLang.default;

    if (!_.get(bunker.sources, lang)) {
        lang = config.bunkerLang.default;
    }

    return _.get(bunker.sources, [lang].concat(path), defaultValue);
}

/**
 * Фильтруя поля переданного объекта, возвращает массив объектов, содержащих данные "этажей",
 * которые должны быть отображены на странице.
 * @param {Object} pageData Объект с данными страницы.
 * @param {Function | null} [additionalFilter] Функция-фильтр, получающая на вход объект
 * с данными одного "этажа" и возвращающая true, если данный "этаж" должен быть отображён на
 * странице, или false - если не должен.
 * Если данный параметр не задан, то в качестве фильтра используется функция,
 * всегда возвращающая true.
 * @returns {Object[]}
 */
function getPageLevels(pageData, additionalFilter) {
    additionalFilter = additionalFilter || (() => true);

    return _(pageData)
        .values()
        .remove(level => level && level.enabled && _.has(level, 'type') && additionalFilter(level))
        .sortBy('order')
        .value();
}

function getImageSize(orig, size) {
    return orig && orig.replace(/\/orig$/, `/${size}`);
}

function replaceTemplates(value, data) {
    var rx = /{{([\w.]+)}}/g;

    return value
        .replace(rx, (str, arg) => _.get(data, arg, ''))
        .replace(/&nbsp;/g, ' ');
}

function escapeName(object) {
    if (object.hasOwnProperty('firstname') && object.hasOwnProperty('lastname')) {
        object.firstname = _.escape(object.firstname);
        object.lastname = _.escape(object.lastname);
    }

    return object;
}

function getMessageTexts(uatraits, texts) {
    const { browsersTexts } = texts;
    const foundTexts = getBrowserSpecificData({ uatraits, data: browsersTexts });

    return foundTexts ? {
        info: foundTexts.info,
        alert: foundTexts.alert
    } : {
        info: texts.info,
        alert: texts.alert
    };
}

/**
 * Удаляет слэш в конце урла
 * @param {String} path
 * @returns {String}
 */
function removeLastSlash(path) {
    return path.replace(/\/$/, '');
}

/**
 * Функция, которая объединяет данные для переключателя локалей
 * Объединяет данные из поля 'regions' текущей модели и значения по умолчанию
 * data.regions заполняется в Бункере для каждой из страниц
 * @returns {Array}
 */
function getRegions() {
    return config.regionsOrder.map(regionId => {
        const pathname = buildPath('/');

        return _.assign(config.regions[regionId], {
            url: `${config.host}${regionId}${pathname}`,
            tld: regionId
        });
    });
}

/**
 * Функция, которая проверяет, что браузер пользователя является валидным.
 * Берет имя и версию браузера пользователя из uatraits,
 * проходит по массиву допустимых браузеров и ищет совпадение по имени
 * если есть совпадение по имени - происходит проверка по версии
 * @param {Object} uatraits Объект данных о браузере.
 * @param {Array} [availableBrowsers] Массив допустимых браузеров
 * @returns {Boolean}
 */
function isValidBrowser(uatraits, availableBrowsers) {
    if (!uatraits || !uatraits.BrowserVersion || !uatraits.BrowserName) {
        return false;
    }

    const name = uatraits.BrowserName;
    const versionParts = uatraits.BrowserVersion.split('.');

    for (let i = 0; i < availableBrowsers.length; i++) {
        const browser = availableBrowsers[i];
        const nameRegex = new RegExp(browser.name, 'i');

        if (nameRegex.test(name)) {
            const requiredVersionParts = browser.version.split('.');
            const minVersionLength = _.min([requiredVersionParts.length, versionParts.length]);

            for (let j = 0; j < minVersionLength; j++) {
                if (Number(versionParts[j]) > Number(requiredVersionParts[j])) {
                    return true;
                } else if (Number(versionParts[j]) < Number(requiredVersionParts[j])) {
                    return false;
                }
            }

            return true;
        }
    }

    return false;
}

/**
 * Получает данные для футера страницы тестирования с прокторингом
 */
function getProctoringFooter(req, isProctoring) {
    if (!isProctoring) {
        return;
    }

    const proctoringFooter = getBunkerNode(req, ['proctoring-footer']);

    if (!proctoringFooter) {
        throw new Error('Proctoring footer not found');
    }

    return _.assign({
        maxErrorLevel: config.proctoringMaxErrorLevel,
        messages: proctoringFooter.messages.map(data => _.assign({
            texts: getMessageTexts(req.base.uatraits, data.texts)
        }, _.pick(data, ['type', 'limit', 'metric', 'showOptions'])))
    }, _.pick(proctoringFooter, [
        'videoText',
        'audioText',
        'problemsText',
        'supervisorInitError',
        'iframeInitError'
    ]));
}

function getProctoringSettings(req, isProctoring) {
    if (!isProctoring) {
        return;
    }

    const attemptBunkerNode = getBunkerNode(req, 'attempt');
    const proctoringSettings = _.get(attemptBunkerNode, 'proctoringSettings');

    if (!proctoringSettings) {
        throw new Error('Proctoring settings not found');
    }

    const metricSettings = getProctoringMetricSettings(req, proctoringSettings.metricSettings);

    return _.assign({
        metricSettings
    }, _.pick(proctoringSettings, [
        'waitForFinish',
        'fixChecksCount',
        'fixSuccessModal',
        'finishModal',
        'nullifiedFinishModal'
    ]));
}

function getProctoringMetricSettings(req, metricSettings) {
    return metricSettings.map(metricInfo => {
        const { waitModal } = metricInfo;

        const browserWaitModal = getBrowserSpecificData({
            uatraits: req.base.uatraits,
            data: waitModal.browsersSettings
        }) || {};

        return _.assign({
            waitModal: {
                title: browserWaitModal.title || waitModal.title,
                content: browserWaitModal.content || waitModal.content
            }
        }, _.pick(metricInfo, [
            'metric',
            'criticalValue',
            'criticalDuration',
            'nullify'
        ]));
    });
}

/**
 * Генерирует JWT токен для прокторинга
 */
function createUserJWT(options) {
    const { timeLimit, username, examTitle } = options;

    const examTimeInMinutes = timeLimit / (60 * 1000);
    const now = Date.now();

    const issuedAt = Math.floor(now / 1000);
    const openId = options.openId || uuid(); // unique session ID
    const payload = {
        // JWT validation
        iat: issuedAt,
        exp: issuedAt + 6 * 60 * 60, // expires after 6 hours
        // user fields
        username,
        // room fields
        id: openId,
        subject: examTitle,
        timeout: examTimeInMinutes // session duration in minutes
    };
    const userProctoringToken = jwt.sign(payload, config.proctoringJwtSecret);

    return { userProctoringToken, openId };
}

/**
 * Получает данные для текущей ОС и браузера пользователя
 * @param options
 * @returns {*}
 */
function getBrowserSpecificData(options) {
    const { uatraits, data } = options;

    const osFamily = _.get(uatraits, 'OSFamily', 'Windows');
    const isAllowedOs = _.includes(['MacOS', 'Windows'], osFamily);
    const os = isAllowedOs ? osFamily : 'Windows';
    const browser = _.get(uatraits, 'BrowserBase') || _.get(uatraits, 'BrowserName', 'Chromium');

    return _.find(data, item => {
        const browserRegex = new RegExp(item.browser, 'i');
        const osRegex = new RegExp(item.os, 'i');

        return osRegex.test(os) && browserRegex.test(browser);
    });
}

function isExamAllowed(slug) {
    const allowedExamsParam = process.env.ALLOWED_EXAMS || '';

    const allowedExams = _(allowedExamsParam)
        .split(' ')
        .compact()
        .value();

    return _.isEmpty(allowedExams) || allowedExams.includes(slug);
}

function getMaterials(req, { examSlug, sections = [], lastSource }) {
    const materialCode = config.examSlugToMaterialCode[examSlug];
    const materials = getBunkerNode(req, ['materials', materialCode], { sections: [] });
    const examSettings = getBunkerNode(req, ['exams', examSlug]);
    const alwaysShowMaterials = _.get(examSettings, 'alwaysShowMaterials', false);
    const failedSectionsCodes = sections
        .filter(section => !section.passed)
        .map(section => section.code);

    const materialsSections = lastSource === 'crit-metrics' ? [] : materials.sections;
    const sectionsToRender = alwaysShowMaterials ?
        materialsSections :
        materialsSections.filter(materialSection => {
            return failedSectionsCodes.includes(materialSection.anchor);
        });

    return {
        title: materials.title,
        sections: sectionsToRender
    };
}

/**
 * Строит строку параметров URL вида '?p1=v1&pn=vn' из объекта. Возвращает пустую строку если объект пустой
 * @param {Object} params
 * @returns {String}
 */
function buildQueryStr(params) {
    const paramStrings = [];
    for (const k of Object.keys(params)) {
        if (params[k]) {
            paramStrings.push(`${encodeURIComponent(k)}=${encodeURIComponent(params[k])}`);
        }
    }
    return paramStrings.length ? `?${paramStrings.join('&')}` : '';
}

module.exports = {
    getBunkerNode,
    getPageLevels,
    getImageSize,
    replaceTemplates,
    escapeName,
    removeLastSlash,
    getRegions,
    isValidBrowser,
    getProctoringFooter,
    getProctoringSettings,
    createUserJWT,
    getBrowserSpecificData,
    isExamAllowed,
    getMaterials,
    buildQueryStr
};
