'use strict';
const _ = require('lodash');
const config = require('cfg');
const url = require('url');
const buildPath = require('../lib/build-path');
const helper = require('../lib/helper');
const environment = require('yandex-environment');

function getUserInfo(req) {
    const blackbox = req.blackbox;

    return {
        login: _.get(blackbox, 'raw.login'),
        uid: blackbox.uid,
        yandexuid: req.cookies.yandexuid
    };
}

/**
 * Функция, возвращающая URL текущей страницы
 * @param {Object} req
 * @param {Boolean} trimQueryString - обрезать ли query-параметры (после знака '?')
 * @param {String[]} leaveQueryParams - имена параметров которые не будут обрезаны
 * @returns {String}
 */
function getCurrentUrl(req, trimQueryString, leaveQueryParams) {
    let pathname;

    try { // Оберегаем себя от ошибки 'URIError: URI malformed'
        pathname = decodeURIComponent(req.originalUrl);
    } catch (e) {
        pathname = buildPath('/');
    }

    let urlParsed = new URL(pathname, `https://${req.headers.host}`);
    const query = {};
    for (let p of leaveQueryParams || []) {
        query[p] = urlParsed.searchParams.get(p);
    }

    return url.format({
        protocol: 'https',
        host: req.headers.host,
        pathname: urlParsed.pathname,
        search: trimQueryString ? helper.buildQueryStr(query) : urlParsed.search
    });
}

module.exports = function expressBase() {
    return function (req, res, next) {
        req.base = {
            host: req.headers.host,
            homeUrl: url.format({
                protocol: 'https',
                host: req.headers.host,
                pathname: buildPath('/')
            }),
            staticHost: config.static.host,
            currentUrl: getCurrentUrl(req),
            retpath: getCurrentUrl(req, true, ['retlink']),
            mordaHost: url.format({
                protocol: 'https',
                host: config.morda[req.tld]
            }),
            faviconUrl: buildPath('favicon.ico'),
            debug: ['local', 'testing'].indexOf(environment) > -1 && req.query.debug,
            language: req.language,
            contentRegion: req.tld.split('.').pop(),
            tld: req.tld,
            uatraits: req.uatraits,
            blackbox: req.blackbox,
            userInfo: getUserInfo(req),
            nonce: req.nonce,
            secretkey: req.secretkey,
            shareUrl: config.share.lib,
            socialList: config.share.socialList,
            subscribeFormPath: config.forms.subscribeUrl,
            passportProfile: url.format({
                protocol: config.passport.protocol,
                host: config.passport.host(req.tld),
                pathname: 'profile'
            }),
            passportHost: url.format({
                protocol: config.passport.protocol,
                host: config.passport.host(req.tld)
            }),
            forms: config.forms,
            metrikaCounters: config.metrikaCounters,
            settings: helper.getBunkerNode(req, 'settings', {}),
            regions: helper.getRegions(),
            raven: {
                config: require('../lib/get-raven-config'),
                dsn: config.sentry.public
            },
            logUrl: buildPath.ajax('log')
        };

        next();
    };
};
