'use strict';

const _ = require('lodash');
const buildPath = require('../lib/build-path');

const BaseModel = require('./base');

class Attempt extends BaseModel {
    get attemptId() {
        return _.get(this._req, 'params.attemptId');
    }

    get examSlug() {
        return _.get(this._req, 'params.examSlug');
    }

    get retlink() {
        return _.get(this._req, 'query.retlink');
    }

    fetch() {
        var options = {
            url: this._path('/attempt', this.attemptId, 'nextQuestion')
        };

        return this._request(options)
            .bind(this)
            .then(this._sanitizeQuestionData)
            .then(this._addAttemptInfo);
    }

    create() {
        var { openId } = this._req.query;

        var options = {
            method: 'POST',
            body: { openId },
            url: this._path('/attempt', this.examSlug)
        };

        return this._request(options);
    }

    answer() {
        var options = {
            method: 'POST',
            body: this._req.body,
            url: '/attempt/answer'
        };

        return this._request(options);
    }

    finish(isCritMetrics, requestNullify) {
        var body = {};

        if (isCritMetrics) {
            body.isCritMetrics = isCritMetrics;
        }

        if (requestNullify) {
            body.requestNullify = requestNullify;
        }

        var options = {
            method: 'POST',
            body: body,
            url: this._path('/attempt', this.attemptId, 'finish')
        };

        return this._request(options);
    }

    getProctoringMetrics() {
        var options = {
            url: this._path('/attempt', this.attemptId, 'proctoringMetrics')
        };

        return this._request(options);
    }

    saveProctoringMetrics() {
        var options = {
            method: 'POST',
            body: this._req.body,
            url: this._path('/attempt', this.attemptId, 'proctoringMetrics')
        };

        return this._request(options);
    }

    _sanitize(text) {
        // В текстах вопросов и ответов обрабатываем все теги кроме <br>
        if (!_.isString(text)) {
            return text;
        }

        return text
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/&lt;br\/?&gt;/g, '<br>');
    }

    _sanitizeQuestionData() {
        if (_.isEmpty(this._data)) {
            return this;
        }

        _.update(this._data, 'text', this._sanitize);

        this._data.answers.forEach((answer, index) => {
            _.update(this._data, ['answers', index, 'text'], this._sanitize);
        });

        return this;
    }

    _addAttemptInfo() {
        this._data.attemptId = this.attemptId;

        this._data.resultUrl = buildPath('exam', this.examSlug, this.attemptId, 'result');

        this._data.nextUrl = buildPath.ajax('exam', this.examSlug, this.attemptId, 'next');
        this._data.answerUrl = buildPath.ajax('exam', this.examSlug, this.attemptId, 'answer');
        this._data.finishUrl = buildPath.ajax('exam', this.examSlug, this.attemptId, 'finish');
        this._data.proctoringMetricsUrl = buildPath.ajax('exam', this.examSlug, this.attemptId, 'proctoringMetrics');
        this._data.pingUrl = buildPath.ajax('ping');

        this._data.retlink = this.retlink;

        return this;
    }
}

module.exports = Attempt;
