'use strict';

const request = require('../request');
const _ = require('lodash');
const http = require('http');
const config = require('cfg');
const keepAliveAgent = new http.Agent({ keepAlive: true, maxSockets: 1024 });
const getImageSize = require('../lib/helper').getImageSize;
const path = require('path');

class Base {
    constructor(req, attrs, data) {
        this._req = req;
        this._attrs = attrs;
        this._ip = req.ip;
        this._userhost = req.headers.host;
        this._cookie = req.headers.cookie;
        this._data = data;
    }

    /**
     * Получение данных модели
     *
     * @return {Object}
     */
    toJSON() {
        return this._data;
    }

    /**
     * Функция склеивает переданные ей аргументы через символ "/".
     * Каждый аргумент приводится к строке.
     *
     * @private
     */
    _path() {
        var args = Array.prototype.slice.call(arguments);
        args = args.map(arg => {
            if (_.isString(arg)) {
                return arg;
            }

            if (_.isInteger(arg)) {
                return arg.toString();
            }

            throw new Error('Argument must be a string or a number');
        });

        return path.join.apply(path, args);
    }

    _removeEmptyProperties(obj) {
        for (var key in obj) {
            if (obj[key] === undefined) {
                delete obj[key];
            }
        }
        return obj;
    }

    /**
     * Запрос на бекенд
     * @param {Object} options параметры запроса к бекенду,
     *                  @see https://www.npmjs.com/package/request#request-options-callback
     * @returns {Promise.<TResult>}
     * @private
     */
    _request(options) {
        options = _.assign({
            agent: keepAliveAgent,
            retries: 0,
            json: true
        }, options);
        var headers = _.assign({
            userhost: this._userhost,
            userip: this._ip,
            cookie: this._cookie,
            'accept-encoding': 'gzip',
            'content-type': 'application/json',
            requestId: this._req.id
        }, options.headers);

        options.headers = this._removeEmptyProperties(headers);
        options.method = options.method || 'GET';
        if (options.body) {
            options.body = JSON.stringify(options.body);
        }
        options.query = this._removeEmptyProperties(options.query);

        var url = `${config.api.host}/${config.api.version}${options.url}`;
        delete options.url;

        return request(url, options)
            .bind(this)
            .then(this._onResponse)
            .then(this._parseData);
    }

    /**
     * Преобразование данных модели
     * @returns {Base}
     * @private
     */
    _parseData() {
        return this;
    }

    /**
     * Обрабатка ответа от бекенда
     * @return {Base}
     * @private
     */
    _onResponse(data) {
        this._data = data.body;
        return this;
    }

    _resizeCertImage(cert) {
        return _(cert)
            .set('image', {
                orig: cert.imagePath,
                w600: getImageSize(cert.imagePath, 'w600')
            })
            .omit(cert, 'imagePath')
            .value();
    }
}

module.exports = Base;
