'use strict';

const _ = require('lodash');
const url = require('url');
const config = require('cfg');

const buildPath = require('../lib/build-path');
const { escapeName, getBunkerNode, isExamAllowed, getMaterials } = require('../lib/helper');

const BaseModel = require('./base');

class Certificates extends BaseModel {
    fetchPublic() {
        const options = {
            url: this._path('/certificates', this._req.params.hashedUserId)
        };
        return this._request(options)
            .bind(this)
            .then(this._escapeName)
            .then(this._setSelectedCert)
            .then(this._groupCerts);
    }

    fetchMy() {
        const formattedUrl = url.format({
            pathname: '/certificates/my',
            query: {
                exams: Object.keys(config.examSlugToServiceCode)
            }
        });

        return this._request({
            url: formattedUrl
        })
            .bind(this)
            .then(data => {
                // TODO: доработать, когда появится прокторинг на com
                if (this._req.tld !== 'ru') {
                    this._data.pendingTrials = [];
                }

                return data;
            })
            .then(this._escapeName)
            .then(this._addBadges)
            .then(this._groupCerts)
            .then(this._filterFailedTrialsByAllowedExams)
            .then(this._prepareFailedTrials)
            .then(this._sortPendingTrials)
            .then(this._addAdditionalCertData)
            .then(this._addPublicPageUrl);
    }

    _escapeName() {
        this._data = escapeName(this._data);
        this._data.certificates = this._data.certificates.map(escapeName);

        return this;
    }

    /**
     * Если на публичной странице сертификатов в URL был передам параметр `id`,
     * то кладем найденный сертификат в data.content (для динамического og:title)
     * @returns {Object}
     * @private
     */
    _setSelectedCert() {
        const certId = _.get(this._req, 'query.id');

        if (certId) {
            this._data.selectedCert = _.find(this._data.certificates, { certId: parseInt(certId, 10) });
        }

        return this;
    }

    /**
     * Добавляем каждому сертификату ссылку на его страницу и ссылку на начало теста,
     * группируем список сертификатов по свойству 'activity'
     * @returns {Object}
     * @private
     */
    _groupCerts() {
        if (_.isEmpty(this._data.certificates)) {
            this._data.noCerts = true;
        }

        this._data.certificates = _(this._data.certificates)
            .map(cert => this._resizeCertImage(cert))
            // Разбиваем список на сертификаты и ачивки { cert: [], achievement: [] }
            .groupBy('certType')
            .defaults({ cert: [], achievement: [] })
            .value();

        this._data.certificates.cert = _(this._data.certificates.cert)
            .map(cert => {
                // Ссылка ведет на страницу проверки сертификата с заполненными query-параметрами
                cert.certUrl = url.format({
                    protocol: 'https',
                    host: this._req.base.host,
                    pathname: buildPath('certificates'),
                    query: _.pick(cert, ['certId', 'lastname'])
                });

                // Проверяем валидность сертификата, если просрочен, то считаем его inactive
                var overDue = new Date(cert.dueDate) < new Date();
                cert.active = !cert.active || overDue ? 'inactive' : 'active';

                return cert;
            })
            // Сортируем сертификаты по хронологии
            .orderBy(cert => new Date(cert.confirmedDate), ['desc'])
            // Разбиваем список сертификатов на активные и неактивные { active: [], inactive: [] }
            .groupBy('active')
            .defaults({ active: [], inactive: [] })
            .value();

        this._data.certificates.achievement = _(this._data.certificates.achievement)
            .uniqBy('exam.id')
            .filter('previewImagePath')
            .map(achievement => {
                achievement.achievementUrl = url.format({
                    protocol: 'https',
                    host: this._req.base.host,
                    pathname: buildPath('certificates', this._data.hashedUserId || this._req.params.hashedUserId),
                    query: { id: _.get(achievement, 'certId') }
                });

                achievement.isHiddenExam = config.hiddenExams.includes(achievement.exam.slug);

                return achievement;
            })
            .value();

        return this;
    }

    _filterFailedTrialsByAllowedExams() {
        this._data.failedTrials = this._data.failedTrials.filter(trial => {
            return isExamAllowed(trial.examSlug);
        });
    }

    _prepareFailedTrials() {
        this._data.failedTrials = _(this._data.failedTrials)
            .orderBy(trial => new Date(trial.started), ['desc'])
            .map(trial => {
                const { examSlug, trialId, sections, source } = trial;
                const tld = config.tldForExamSlugs[examSlug];
                const host = this._buildHostWithTld(this._req.base.host, tld);

                trial.materials = getMaterials(this._req, { examSlug, sections, lastSource: source });

                trial.resultUrl = url.format({
                    protocol: 'https',
                    host: host,
                    pathname: buildPath('exam', examSlug, trialId, 'result')
                });

                trial.examUrl = url.format({
                    protocol: 'https',
                    host: host,
                    pathname: buildPath('exam', examSlug)
                });

                return trial;
            })
            .value();

        return this;
    }

    _sortPendingTrials() {
        this._data.pendingTrials = _(this._data.pendingTrials)
            .orderBy(trial => new Date(trial.started), ['desc']).value();

        return this;
    }

    _addAdditionalCertData() {
        this._data.certificates.cert.active.forEach(cert => {
            // Ссылка на начало нового тестирования (если есть)
            const examSlug = _.get(cert, 'exam.slug');
            const tld = config.tldForExamSlugs[examSlug];
            const host = this._buildHostWithTld(this._req.base.host, tld);

            cert.attemptUrl = examSlug && url.format({
                protocol: 'https',
                host: host,
                pathname: buildPath('exam', examSlug)
            });
            cert.downloadUrl = buildPath('certificate', 'pdf', this._data.hashedUserId, cert.certId);

            // Является ли тест открытым к прохождению
            cert.isExamAllowed = isExamAllowed(examSlug);
        });

        return this;
    }

    _addPublicPageUrl() {
        this._data.publicPageUrl = buildPath('certificates', this._data.hashedUserId);
        return this;
    }

    _addBadges() {
        const {
            badges,
            badgesSizes,
            certPopups,
            certPopupSize,
            yaLetterCertificationDate,
            certLangs
        } = config;
        const { badgeCode } = getBunkerNode(this._req, 'settings');

        this._data.certificates.forEach(certificate => {
            const serviceCode = certificate.service.code;
            const confirmedDate = new Date(certificate.confirmedDate);
            const isYaLetterCert = yaLetterCertificationDate < confirmedDate;
            const certLang = certLangs[serviceCode];

            if (badges.hasOwnProperty(serviceCode)) {
                _.assign(certificate, {
                    badges: badges[serviceCode],
                    badgeCode: badgeCode,
                    badgesSizes: badgesSizes,
                    examSlugToServiceCode: config.examSlugToServiceCode,
                    certPopupSrc: isYaLetterCert && certPopups[certLang],
                    certPopupSize: isYaLetterCert && certPopupSize,
                    isYaLetterCert: isYaLetterCert
                });
            }
        });

        return this;
    }

    _buildHostWithTld(baseHost, tld) {
        const parsedHost = baseHost.split('.');
        parsedHost[parsedHost.length - 1] = tld;

        return parsedHost.join('.');
    }
}

module.exports = Certificates;
