'use strict';
const config = require('cfg');
const log = require('./logger');
const proxy = require('express-http-proxy');

const router = require('./lib/router-helper');
const secretkeyCheck = require('./middleware/secretkey-check');
const validateExamId = require('./middleware/validate-exam-id');
const validateExamLang = require('./middleware/validate-exam-language');

const auth = require('./controllers/auth');
const attempt = require('./controllers/attempt');
const exam = require('./controllers/exam');
const examCluster = require('./controllers/exam-cluster');
const index = require('./controllers/index');
const identificationSteps = require('./controllers/identificationSteps');
const attemptResult = require('./controllers/attempt-result');
const myCertificates = require('./controllers/my-certificates');
const publicCertificates = require('./controllers/public-certificates');
const certificatesSearch = require('./controllers/certificates-search');
const materials = require('./controllers/materials');
const faq = require('./controllers/faq');
const usage = require('./controllers/usage');
const ping = require('./controllers/ping');
const proRules = require('./controllers/proRules');
const logData = require('./controllers/log-data');
const screenShare = require('./controllers/screen-share');
const cameraMicrophone = require('./controllers/camera-microphone');

module.exports = function Router(app) {
    router
        .get(app, '/', index, 'indexPage')
        .get(app, '/auth', auth, 'authPage')
        .get(app, '/exam/:examSlug/start', attempt, 'create', { middleware: [validateExamId, validateExamLang] })
        .get(app, '/exam/:examSlug/go', identificationSteps, 'go', { middleware: [validateExamId, validateExamLang] })
        .get(app, '/exam/:examSlug/:attemptId/result', attemptResult, 'resultPage', { middleware: [validateExamId, validateExamLang] })
        .get(app, '/exam/:examSlug/:attemptId/next', attempt, 'nextQuestion', { middleware: [validateExamId, validateExamLang], ajax: true })
        .get(app, '/exam/:examSlug/:attemptId', attempt, 'attemptPage', { middleware: [validateExamId, validateExamLang] })
        .get(app, '/exam/:examSlug', exam, 'examPage', { middleware: [validateExamId, validateExamLang] })
        .get(app, '/certificates/my', myCertificates, 'myCertificatesPage')
        .get(app, '/certificates/:hashedUserId', publicCertificates, 'publicCertificatesPage')
        .get(app, '/certificates', certificatesSearch, 'certificatesSearchPage')
        .get(app, '/materials/:code', materials, 'materialsPage')
        .get(app, '/faq', faq, 'contentPage')
        .get(app, '/usage', usage, 'usagePage')
        .get(app, '/ping', ping, 'ping')
        .get(app, '/pro-rules', proRules, 'contentPage')
        .get(app, '/screen-share', screenShare, 'contentPage')
        .get(app, '/camera-microphone', cameraMicrophone, 'contentPage')
        .get(app, '/:clusterSlug', examCluster, 'clusterPage')

        .post(app, '/exam/:examSlug/identificate', identificationSteps, 'identificate', { middleware: [secretkeyCheck, validateExamId, validateExamLang], ajax: true })
        .post(app, '/exam/:examSlug/:attemptId/answer', attempt, 'answer', { middleware: [secretkeyCheck, validateExamId, validateExamLang], ajax: true })
        .post(app, '/exam/:examSlug/:attemptId/finish', attempt, 'finish', { middleware: [secretkeyCheck, validateExamId, validateExamLang], ajax: true })
        .post(app, '/exam/:examSlug/:attemptId/proctoringMetrics', attempt, 'saveProctoringMetrics', { middleware: [secretkeyCheck, validateExamId, validateExamLang], ajax: true })
        .post(app, '/certificates/check', certificatesSearch, 'check', { middleware: secretkeyCheck, ajax: true })
        .post(app, '/log', logData, 'logBody', { ajax: true });

    app.use(`${config.router.namespace}/certificate/pdf/:hashedUserId/:certId`, proxy(config.api.host, {
        proxyReqPathResolver: function (req) {
            return `/${config.api.version}/certificate/pdf/${req.params.hashedUserId}/${req.params.certId}`;
        },
        userResDecorator: function (proxyRes, proxyResData) {
            if (proxyRes.statusCode === 200) {
                return proxyResData;
            }

            throw new Error(proxyResData.toString('utf-8'));
        },
        proxyErrorHandler: function (err, res) {
            log.error(err);
            res.sendStatus(404);
        }
    }));
};
