/**
 * COPY & PASTE https://github.yandex-team.ru/Afisha/afisha-frontend/blob/master/static/.enb/enb/techs/gemini.js
 * gemini
 * ======
 *
 * Склеивает *gemini*-файлы по deps'ам в виде `?.gemini.js`.
 * Предназначен для сборки gemini тестов.
 *
 * **Опции**
 *
 * * *String* **target** — Результирующий таргет. По умолчанию — `?.gemini.js`.
 * * *String* **dirsTarget** — dirs-таргет, на основе которого получается список `?.priv-gemini.js` файлов (его предоставляет технология `files`). По умолчанию — `?.dirs`.
 * * *String* **blocksPath** —  Путь к диретории html файлов для тестов.
 * * *String* **techSuffix** — Cуфикс целевых прив-файлов. По умолчанию — `gemini.js`.
 * * *String* **blockPrefix** — Префикс для прив-файлов с тестами. По умолчанию — `gemini-`.
 *
 * **Пример**
 *
 * ```javascript
 * nodeConfig.addTech(require('enb/techs/priv-gemini'), {
 *     blocksPath: nodeConfig.resolvePath(path.join('desktop.gemini', 'html'))
 * });
 * ```
 */
var path = require('path');
var Vow = require('vow');
var vfs = require('enb/lib/fs/async-fs');

module.exports = require('enb/lib/build-flow').create()
    .name('gemini')
    .target('target', '?.gemini.js')
    .defineRequiredOption('blocksPath')
    .defineOption('techSuffix', 'gemini.js')
    .defineOption('blockPrefix', 'gemini-')
    .useDirList('gemini')
    .builder(function (dirs) {
        var node = this.node;
        var blocksPath = this._blocksPath;
        var techSuffix = this._techSuffix;
        var blockPrefix = this._blockPrefix;
        var rootUrl = path.parse(blocksPath).name;
        var suites = {};

        var promises = dirs.map(function (dir) {
            return getDirContent(node, dir, techSuffix).then(function (files) {
                files.forEach(function (file) {
                    if (!file) {
                        return;
                    }

                    var suite = path.basename(file.name, '.' + file.suffix);

                    if (!suites[suite]) {
                        suites[suite] = [];
                    }

                    suites[suite].push(file.content);
                });
            });
        });

        return Vow.all(promises).then(function () {
            return renderFile(suites, rootUrl, blockPrefix);
        });
    })
    .createTech();

/**
 * Считываем все тесты из директории
 * @param {Object} node
 * @param {String} dir
 * @param {string} techSuffix
 * @returns {Promise<Object[]>}
 */
function getDirContent(node, dir, techSuffix) {
    return Vow.all(dir.files.map(function (file) {
        if (file.suffix !== techSuffix) {
            return;
        }

        return getFileContent(node, file.fullname).then(function (content) {
            return {
                name: file.fullname,
                suffix: file.suffix,
                content: content
            };
        });
    }));
}

/**
 * Получаем код одного теста
 * @param {Object} node
 * @param {String} file
 * @returns {Promise<String>}
 */
function getFileContent(node, file) {
    return vfs.read(file, 'utf8').then(function (data) {
        var relPath = node.relativePath(file);

        return [
            '// begin: ' + relPath,
            '(function () {',
            data,
            '}());',
            '// end: ' + relPath
        ].join('\n');
    });
}

/**
 * Генерируем gemini файл со всеми тестами
 * @param {Object} suites
 * @param {String} rootUrl
 * @param {String} blockPrefix
 * @returns {String}
 */
function renderFile(suites, rootUrl, blockPrefix) {
    return [
        Object.keys(suites).map(function (suite) {
            var suiteData = suites[suite].join('\n');
            return renderSuite(suite, suiteData, rootUrl, blockPrefix);
        }).join('\n')
    ].join('');
}

/**
 * Генерируем один gemini тест
 * @param {String} name Название теста
 * @param {String} data Код теста
 * @param {String} rootUrl
 * @param {String} blockPrefix
 * @returns {String}
 */
function renderSuite(name, data, rootUrl, blockPrefix) {
    var blockName = blockPrefix + name;
    var url = [rootUrl, name, name + '.html'].join('/');

    var str = [
        '// begin suite: {{suite}}',
        '(function () {',
        '  gemini.suite("{{suite}}", function (suite) {',
        '    var blockName = "' + blockName + '";',
        '    suite = suite',
        '      .setUrl("/{{url}}")',
        '      .setCaptureElements("." + blockName);',
        '',
        data,
        '  });',
        '}());',
        '// end suite: {{suite}}'
    ].join('\n');

    return str
        .split('{{suite}}').join(name)
        .split('{{url}}').join(url)
        .split('{{block}}').join(blockName);
}
