/**
 * priv-gemini
 * ===========
 *
 * Генерирует набор статичных html файлов на основе *priv-gemini*-файлы.
 * Предназначен для сборки статичных страниц для тестирования скриншотами.
 *
 * **Опции**
 *
 * * *String* **target** — Результирующий таргет. По умолчанию — `?.priv.gemimi.js`. Будет содержать список сгенерированных html файлов.
 * * *String* **privFile** — Абсолютный путь к собранному `.priv.js` файлу. По умолчанию — `?.priv.js`.
 * * *String* **bhFile** — Абсолютный путь к собранному `.bh.js` файлу. По умолчанию — `?.bh.js`.
 * * *String* **destPath** — Путь к диретории результирующих html файлов.
 * * *String* **dirsTarget** — dirs-таргет, на основе которого получается список `?.priv-gemini.js` файлов (его предоставляет технология `files`). По умолчанию — `?.dirs`.
 * * *String* **pageBlockName** — Название блока для оборачивания всех тестируемых привов.
 * * *String* **techSuffix** — Cуфикс целевых прив-файлов. По умолчанию — `priv-gemini.js`.
 * * *String* **blockPrefix** — Префикс для прив-файлов с тестами. По умолчанию — `gemini-`.
 *
 * **Пример**
 *
 * ```javascript
 * nodeConfig.addTech(require('enb/techs/priv-gemini'), {
 *     destPath: nodeConfig.resolvePath(path.join('desktop.gemini', 'html'))
 * });
 * ```
 */
var path = require('path'),
    Vow = require('vow'),
    vfs = require('enb/lib/fs/async-fs');
module.exports = require('enb/lib/build-flow').create()
    .name('priv-gemini')
    .target('target', '?.priv.gemini.js')
    .defineRequiredOption('privFile')
    .defineRequiredOption('bhFile')
    .defineRequiredOption('destPath')
    .defineOption('pageBlockName')
    .defineOption('techSuffix', 'priv-gemini.js')
    .defineOption('blockPrefix', 'gemini-')
    .useSourceFilename('privFile', '?.priv.js')
    .useSourceFilename('bhFile', '?.bh.js')
    .useDirList('gemini')
    .builder(function (privFile, bhFile, dirs) {
        var node = this.node,
            blocks = require(privFile),
            bh = require(bhFile),
            destPath = node.resolvePath(this._destPath),
            pageBlockName = this._pageBlockName,
            techSuffix = this._techSuffix,
            blockPrefix = this._blockPrefix;

        var blockNames = getBlockNames(blocks, dirs, techSuffix, blockPrefix);

        try {
            var promises = blockNames.map(function (blockName) {
                var bemjson = renderBlock(blocks, blockName, pageBlockName),
                    html = bh.apply(bemjson),
                    testName = blockName.slice(blockPrefix.length),
                    dirName = path.join(destPath, testName),
                    fileName = path.join(dirName, testName + '.html');

                return vfs.makeDir(dirName).then(function () {
                    return saveHtml(fileName, html);
                });
            });
        } catch (err) {
            console.error(err);
            // todo: ENB 0.17 не ловит rejected промисы...
            process.exit(1);
            return Vow.reject(err);
        }

        return Vow.all(promises)
            .then(function (files) {
                return '/* ' + files.join('*/\n/*') + ' */';
            })
            .catch(function (err) {
                console.error(err);
                // todo: ENB 0.17 не ловит rejected промисы...
                process.exit(1);
                throw err;
            });
    })
    .createTech();

/**
 * Получаем из списка директорий список блоков для тестирования
 * @param {Blocks} blocks
 * @param {String[]} dirs
 * @param {String} techSuffix
 * @param {String} blockPrefix
 * @returns {String[]}
 */
function getBlockNames(blocks, dirs, techSuffix, blockPrefix) {
    return dirs.reduce(function (blockNames, dir) {
        return dir.files.reduce(function (blockNames, file) {
            if (file.suffix !== techSuffix) {
                return blockNames;
            }

            var testName = path.basename(file.name, '.' + file.suffix),
                blockName = blockPrefix + testName;

            // register gemini priv
            require(file.fullname)(blocks, blockName);

            if (blockNames.indexOf(blockName) === -1) {
                blockNames.push(blockName);
            }

            return blockNames;
        }, blockNames);
    }, []);
}

/**
 * @param {Blocks} blocks
 * @param {String} blockName
 * @param {String} [pageBlockName]
 * @returns {Object}
 */
function renderBlock(blocks, blockName, pageBlockName) {
    if (pageBlockName) {
        return blocks.exec(pageBlockName, {
            title: 'Test page for ' + blockName,
            blockName: blockName
        });
    } else {
        return blocks.exec(blockName);
    }
}

/**
 * @param {String} fileName
 * @param {String} html
 * @returns {Promise<String>}
 */
function saveHtml(fileName, html) {
    vfs.write(fileName, html, 'utf-8').then(function () {
        return node.relativePath(fileName);
    })
}
