module.exports = function (bh) {
    'use strict';

    function getRestrictedMessage(data) {
        if (data.isSpammer) {
            return {
                elem: 'restriction',
                mods: { type: 'spammer' },
                mix: { block: 'html-content' },
                content: bh.lib.i18n('common', 'exam.spammer')
            };
        }

        if (data.isBanned) {
            return {
                elem: 'restriction',
                mods: { type: 'banned' },
                mix: { block: 'html-content' },
                expiredBanDate: data.expiredBanDate
            };
        }

        if (data.isFrozen) {
            return {
                elem: 'restriction',
                mods: { type: 'readonly' },
                mix: { block: 'html-content' },
                freezeFinishTime: data.freezeFinishTime
            };
        }

        if (data.availabilityDate) {
            return {
                elem: 'restriction',
                mods: { type: 'date' },
                mix: { block: 'html-content' },
                timeLeft: data.availabilityDate,
                hasValidCert: data.hasValidCert,
                examType: data.examType
            };
        }

        if (data.isPending) {
            return {
                elem: 'restriction',
                mods: { type: 'pending' },
                mix: { block: 'html-content' },
                content: bh.lib.i18n('common', 'exam.pending')
            };
        }

        if (data.isDisabledForMobile) {
            return {
                elem: 'restriction',
                mix: { block: 'html-content' },
                mods: { type: 'disabled-for-mobile' }
            };
        }

        if (data.isProctoringWithoutSupporting) {
            return {
                elem: 'restriction',
                mods: { type: 'browser' },
                mix: { block: 'html-content' },
                content: bh.lib.i18n('common', 'exam.no-proctoring-supporting')
            };
        }

        if (data.isNotActualLogin) {
            return {
                elem: 'restriction',
                mods: { type: 'not-actual-login' },
                mix: { block: 'html-content' },
                content: bh.lib.i18n('common', 'exam.not-actual-login')
            };
        }

        if (!data.hasLogin) {
            return {
                elem: 'restriction',
                mods: { type: 'no-login' },
                mix: { block: 'html-content' },
                content: bh.lib.i18n('common', 'exam.no-login')
            };
        }

        return {};
    }

    bh.match('exam-start', function (ctx, json) {
        var timeLimit = json.exam.timeLimit && bh.lib.util.date.msToTime(json.exam.timeLimit);
        var state = json.examAccess.state;
        var isFrozen = json.isAuthorized && state === 'frozen';
        var isPending = state === 'pending';
        var isNotActualLogin = state === 'notActualLogin';
        var isDisabledForMobile = json.exam.isProctoring && bh.lib.global.isMobile;
        var isProctoringWithoutSupporting = json.exam.isProctoring &&
            !json.isBrowserSupportProctoring &&
            !bh.lib.global.isMobile;
        var isSpammer = Boolean(json.examAccess.isSpammer);
        var isBanned = state === 'banned';
        var hasLogin = Boolean(json.login);

        var isAvailable = json.isAuthorized &&
            hasLogin &&
            !isSpammer &&
            !isBanned &&
            state === 'enabled' &&
            !isFrozen &&
            !isDisabledForMobile &&
            !isProctoringWithoutSupporting;

        var isButtonDisabled = isSpammer ||
            isBanned ||
            !hasLogin ||
            state === 'disabled' ||
            state === 'frozen' ||
            isPending ||
            isDisabledForMobile ||
            isProctoringWithoutSupporting ||
            isNotActualLogin;

        ctx
            .js({
                isMobile: bh.lib.global.isMobile,
                isAuthorized: json.isAuthorized
            })
            .content([
                timeLimit && {
                    elem: 'time-limit',
                    timeLimit: timeLimit
                },
                json.isAuthorized ? {
                    elem: 'start',
                    disabled: isButtonDisabled,
                    url: state === 'in_progress' ? json.examAccess.attemptUrl : json.exam.attemptUrl,
                    text: bh.lib.i18n('common', (state === 'in_progress') ? 'exam.continue' : 'exam.start')
                } : {
                    elem: 'auth_login',
                    content: [
                        { elem: 'auth' },
                        {
                            elem: 'restriction',
                            mods: { type: 'no-login' },
                            mix: { block: 'html-content' },
                            content: bh.lib.i18n('common', 'exam.no-login')
                        }
                    ]
                },
                isAvailable && [
                    {
                        elem: 'agreement',
                        examAccess: json.examAccess,
                        passportProfile: json.passportProfile
                    },
                    json.terms && json.terms.description && {
                        elem: 'terms-modal',
                        examAccess: json.examAccess,
                        url: json.exam.attemptUrl,
                        terms: json.terms
                    }
                ],
                json.isAuthorized && getRestrictedMessage({
                    isSpammer: isSpammer,
                    isBanned: isBanned,
                    hasLogin: hasLogin,
                    isFrozen: isFrozen,
                    isPending: isPending,
                    isNotActualLogin,
                    availabilityDate: json.examAccess.availabilityDate,
                    expiredBanDate: json.examAccess.expiredDate,
                    isProctoringWithoutSupporting: isProctoringWithoutSupporting,
                    isDisabledForMobile: isDisabledForMobile,
                    freezeFinishTime: json.examAccess.freezeFinishTime,
                    hasValidCert: json.examAccess.hasValidCert,
                    examType: json.exam.type.code
                })
            ]);
    });

    bh.match('exam-start__restriction_type_date', function (ctx, json) {
        var humanDate = bh.lib.util.date.humanizeDate(json.timeLeft);
        var nextAttemptText = bh.lib.i18n('common', 'exam.timeleft', { date: humanDate });
        var hasCertText = bh.lib.i18n('common', 'exam.has-cert');
        var content = json.examType === 'achievement' ?
            {
                elem: 'already-passed',
                content: bh.lib.i18n('common', 'exam.already-passed')
            } :
            [
                json.hasValidCert && {
                    elem: 'has-cert',
                    content: hasCertText
                },
                {
                    elem: 'next-attempt',
                    content: nextAttemptText
                }
            ];

        ctx.content(content);
    });

    bh.match('exam-start__restriction_type_banned', function (ctx, json) {
        var isBannedByExam = Boolean(json.expiredBanDate);
        var type = isBannedByExam ? 'ban-by-exam' : 'superban';
        var humanDate = bh.lib.util.date.humanizeDate(json.expiredBanDate);
        var content = bh.lib.i18n('common', 'exam.' + type, { date: humanDate });

        ctx.content(content);
    });

    bh.match('exam-start__start', function (ctx, json) {
        var isMobile = bh.lib.global.isMobile;

        ctx.content({
            block: 'button2',
            mods: {
                disabled: json.disabled && 'yes',
                theme: 'action',
                width: 'max',
                type: 'link',
                size: isMobile ? 'l' : 'm'
            },
            url: json.url,
            text: json.text
        });
    });

    bh.match('exam-start__agreement', function (ctx, json) {
        var name = json.examAccess.firstname + ' ' + json.examAccess.lastname;

        ctx.content({
            block: 'checkbox',
            mods: { size: 'm' },
            content: {
                elem: 'label',
                content: [
                    bh.lib.i18n('common', 'exam.agreement'),
                    {
                        elem: 'fio',
                        tag: 'strong',
                        content: name.trim()
                    },
                    json.passportProfile && [
                        ' (',
                        {
                            block: 'link',
                            mods: { theme: 'expert' },
                            text: bh.lib.i18n('common', 'exam.agreement.change'),
                            url: json.passportProfile
                        },
                        ')'
                    ]
                ]
            }
        });
    });

    bh.match('exam-start__restriction_type_readonly', function (ctx, json) {
        var delta = new Date(json.freezeFinishTime).getTime() - Date.now();
        var timeLeft = bh.lib.util.date.msToMinutes(delta);
        timeLeft = Math.max(timeLeft, 5);

        ctx.content(bh.lib.i18n('common', 'exam.readonly', { time: timeLeft }));
    });

    bh.match('exam-start__terms-modal', function (ctx, json) {
        var name = json.examAccess.lastname + ' ' + json.examAccess.firstname;
        var terms = json.terms;

        ctx.content([
            {
                block: 'modal',
                mix: { block: 'exam-start', elem: 'terms-modal' },
                mods: {
                    theme: 'normal',
                    visible: 'no',
                    autoclosable: 'yes',
                    type: 'terms'
                },
                content: [
                    {
                        elem: 'close'
                    },
                    {
                        elem: 'terms',
                        content: [
                            {
                                elem: 'terms-texts',
                                content: [
                                    {
                                        elem: 'terms-name',
                                        content: [
                                            {
                                                elem: 'name-explanation',
                                                content: terms.nameExplanation
                                            },
                                            {
                                                elem: 'user-name',
                                                content: name
                                            }
                                        ]
                                    },
                                    {
                                        elem: 'terms-description',
                                        mix: { block: 'html-content' },
                                        content: terms.description
                                    },
                                    {
                                        elem: 'terms-additions',
                                        additions: terms.additions
                                    }
                                ]
                            },
                            {
                                elem: 'terms-clearly',
                                attemptUrl: json.url
                            }
                        ]
                    }
                ]
            }
        ]);
    });
};
