/* global BEM, BH, $ */
BEM.DOM.decl('proctoring-footer', {
    onSetMod: {
        js: {
            inited: function () {
                var hasNotifications = this.params.hasNotifications;

                this.__self._hasNotifications = hasNotifications;
                this.__self._openId = this.params.openId;
                this.__self._attemptId = this.params.attemptId;
                this._report = this.findElem('report');
                this._modal = this.findBlockInside('modal');

                if (hasNotifications) {
                    this._initNotifications();
                }

                this.bindTo(this._report, 'click', this._showReportModal.bind(this));
            }
        }
    },

    /**
     * Инициализация оповещений
     * @private
     */
    _initNotifications: function () {
        this._notifications = [];
        this._popup = this.findBlockInside({
            block: 'popup2',
            modName: 'type',
            modVal: 'notification'
        });
        this._indicators = {
            video: this.findElem('indicator', 'type', 'video'),
            audio: this.findElem('indicator', 'type', 'audio'),
            screen: this.findElem('indicator', 'type', 'screen')
        };

        this._prepareNotificationsPopup();

        this._notificationsElem = this.findElem('notifications');

        this.bindTo(this._notificationsElem, 'click', this._rollNotifications.bind(this));

        this.__self._context = this;
        this.__self._violations = this.params.messages
            .reduce(function (acc, messageInfo) {
                acc[messageInfo.metric] = 0;

                return acc;
            }, {});
    },

    /**
     * Показывает модальное окно "Возникли технические проблемы"
     * @private
     */
    _showReportModal: function () {
        var form = {
            block: 'iframe-form',
            formId: this.params.formId
        };

        if (this.params.attemptId) {
            form.params = [{ key: 'trial_id', value: this.params.attemptId }];
        }

        this._modal.setContent(BH.apply(form));

        this._modal.setMod('visible', 'yes');
    },

    /**
     * Находит метрику по переданному имени
     * @param {String} name
     * @returns {Object | null}
     * @private
     */
    _findMetric: function (name) {
        return this.params.messages.filter(function (metricObj) {
            return metricObj.metric === name;
        })[0];
    },

    /**
     * Удаляет уведомления об ошибках, которые были исправлены
     * @param metrics
     * @private
     */
    _deleteFixedNotifications: function (metrics) {
        var isDeleted = false;

        Object.keys(metrics).forEach(function (metric) {
            var metricInfo = this._findMetric(metric);

            if (metricInfo && metrics[metric] <= metricInfo.limit) {
                var isRemove = this._removeNotification(metricInfo);

                isDeleted = isRemove || isDeleted;
            }
        }, this);

        return isDeleted;
    },

    _removeNotification: function (metric) {
        var notificationIndex = this._notifications.findIndex(function (notification) {
            return notification.type === metric.type && notification.metric === metric.metric;
        });

        if (notificationIndex >= 0) {
            this._notifications.splice(notificationIndex, 1);

            return true;
        }

        return false;
    },

    /**
     * Находит все превышенные метрики
     * @param {Object} metrics
     * @returns {Object | null}
     * @private
     */
    _getOverlimitedMetrics: function (metrics) {
        return Object.keys(metrics).reduce(function (metricsInfo, metric) {
            var metricInfo = this._findMetric(metric);

            if (metricInfo && metrics[metric] > metricInfo.limit) {
                metricsInfo.push(metricInfo);
            }

            return metricsInfo;
        }.bind(this), []);
    },

    _prepareNotificationsPopup: function () {
        this._popup.setAnchor(this);
    },

    _replaceNotifications: function (notifications, violations) {
        var newNotifications = notifications.map(function (metricInfo) {
            var level = metricInfo.showOptions.alert <= violations[metricInfo.metric] ? 'alert' : 'info';
            var text = metricInfo.texts[level];

            return {
                text: text,
                level: level,
                type: metricInfo.type,
                metric: metricInfo.metric
            };
        });

        var isChanged = this._changeNotifications(newNotifications);
        this._renderNotifications(this._notifications);

        return isChanged;
    },

    _renderNotifications: function (notifications) {
        var notificationsElem = BH.apply({
            block: 'proctoring-footer',
            elem: 'notifications-content',
            notifications: notifications
        });

        BEM.DOM.update(this._notificationsElem, notificationsElem);
    },

    _changeNotifications: function (newNotifications) {
        var isChanged = false;
        var oldLength = this._notifications.length;

        newNotifications.forEach(function (newNotification) {
            var notificationIndex = this._notifications.findIndex(function (oldNotification) {
                return newNotification.metric === oldNotification.metric;
            });

            if (notificationIndex === -1) {
                this._notifications.unshift(newNotification);
            } else {
                var isLevelChange = this._notifications[notificationIndex].level !== newNotification.level;
                isChanged = isChanged || isLevelChange;

                this._notifications.splice(notificationIndex, 1, newNotification);
            }
        }, this);

        isChanged = isChanged || oldLength !== this._notifications.length;

        return isChanged;
    },

    /**
     * Переключает уведомления от прокторинга
     * @private
     */
    _rollNotifications: function () {
        var currentNotification = this._notifications.shift();
        this._notifications.push(currentNotification);

        this._renderNotifications(this._notifications);
    },

    /**
     * Меняет цвета индикаторов
     * @param {Array} notifications
     * @private
     */
    _toggleIndicators: function (notifications) {
        ['video', 'audio', 'screen'].forEach(function (indicatorType) {
            var hasViolation = notifications.some(function (metricObj) {
                return metricObj.type === indicatorType;
            });
            var color = hasViolation ? 'red' : 'green';

            this.setMod(this._indicators[indicatorType], 'color', color);
        }, this);
    },

    _togglePopup: function (notifications) {
        if (notifications.length) {
            this._popup.setMod('visible', 'yes');
        } else {
            this._popup.setMod('visible', 'no');
        }
    }
}, {
    /**
     * Показывает оповещение с переданным текстом и иконкой
     * @private
     */
    notify: function (metrics) {
        if (!this._hasNotifications) {
            return;
        }

        var isDeleted = this._context._deleteFixedNotifications(metrics);

        var metricsInfo = this._context._getOverlimitedMetrics(metrics);
        var notifications = [];

        metricsInfo.forEach(function (metricInfo) {
            var violations = this._violations[metricInfo.metric] += 1;
            var isNotify = metricInfo.showOptions.info <= violations;

            if (isNotify) {
                notifications.push(metricInfo);
            }
        }, this);

        this._context._toggleIndicators(notifications);
        var isChanged = this._context._replaceNotifications(notifications, this._violations);

        this._context._togglePopup(this._context._notifications);

        if (isDeleted || isChanged) {
            BH.lib.util.logger.info({
                notifications: this._context._notifications,
                place: 'Change notifications',
                openId: this._openId,
                attemptId: this._attemptId
            });
        }
    },

    /**
     * Выставляет состояние ошибки запуска прокторинга
     * @private
     */
    setProctoringError: function () {
        this._context.setMod('video', 'hidden');
        this._context.setMod('visible', 'yes');
        this._context.setMod(this._context._indicators.video, 'color', 'red');
        this._context.setMod(this._context._indicators.audio, 'color', 'red');
        this._context.setMod(this._context._indicators.screen, 'color', 'red');

        var critAlert = [{
            level: 'alert',
            type: 'crit',
            text: this._context.params.iframeInitError
        }];

        this._context._renderNotifications(critAlert);
        this._context._togglePopup(critAlert);
    },

    /**
     * Меняет состояние при успешной инициализации прокторинга
     * @private
     */
    setProctoringStarted: function () {
        this._context.setMod('visible', 'yes');

        var faceBlock = BEM.blocks['proctoring-face'];
        faceBlock.init();
    },

    /**
     * Скрывает видео пользователя в окошке
     * @private
     */
    hideCameraStream: function () {
        this._context.setMod('video', 'hidden');
    },

    disableNotifications: function () {
        this._context.setMod(this._context._notificationsElem, 'disabled', 'yes');
    },

    enableNotifications: function () {
        this._context.delMod(this._context._notificationsElem, 'disabled');
    }
});
