/// <reference path="./@types/express/index.d.ts" />

declare module 'express-blackbox' {
    import { Request, RequestHandler, Response } from 'express';
    import { GotOptions } from 'got';

    interface Avatar {

        /**
         * Avatar ID (e.g. 1824/4003228553-485655)
         *
         * Can be used to build avatar image url:
         * https://avatars.mds.yandex.net/get-yapic/1824/4003228553-485655/orig
         */
        default: string

        /** Avatar presence flag (false if there are none, true if at least one) */
        empty: boolean
    }

    interface AuthorizationStatus {

        /**
         * Number representation of authorization status:
         * 0 - account exists and active
         * 1 – need to update session cookies
         * 5 – account blocked
         * 6 – account doesn't exist or need to change password
         *
         * See:
         * https://nda.ya.ru/3UNKcC
         */
        id: number

        /**
         * String representation of authorization status:
         * VALID - account exists and active
         * NEED_RESET – need to update session cookies
         * DISABLED – account is blocked
         * INVALID – account doesn't exist or need to change password
         */
        value: string
    }

    interface UserId {

        /** True if the account was registered in the mail, not in the portal */
        hosted: boolean

        /** True if the account was registered using the social networks */
        lite: boolean

        value: string
    }

    interface Karma {

        /**
         * The normalized indicator that the user is a spammer:
         * 0 - not spammer, 100 - sure a spammer.
         */
        value: number
    }

    interface RawKarma {

        /**
         * Indicator that the user is a spammer.
         *
         * More information:
         * https://wiki.yandex-team.ru/passport/karma/
         */
        value: number
    }

    interface DisplayName {

        /** User display name */
        name: string

        /** User avatar */
        avatar: Avatar
    }

    interface PassportAttributes {
        [key: string]: string
    }

    interface UserEmail {

        /** Email address */
        address: string

        /** Email record creation date */
        'born-date': string

        /** True if email is default */
        default: boolean

        /** True if email is used by email collector  */
        rpop: boolean

        /** True if email has one of yandex domain */
        native: boolean

        /** True if notifications disabled for this email */
        silent: boolean

        /** True if email is confirmed unsafely */
        unsafe: boolean

        /** True if email is confirmed */
        validated: boolean
    }

    interface UserAuthId {
        id: string

        /** Authorization cookie creation date */
        time: string

        /** User IP at the moment of authorization */
        ip: string

        /** The number of the machine which generated the cookie */
        host: string
    }

    interface LocationsMap {
        [key: string]: string[]
    }

    interface AliasesMap {
        [key: string]: string
    }

    interface PassportSessionidResponse {
        'address-list'?: UserEmail[]

        /** User age */
        age?: number

        /**
         * User login aliases
         *
         * See:
         * https://wiki.yandex-team.ru/passport/dbmoving/#tipyaliasov
         */
        aliases: AliasesMap

        authid: UserAuthId

        /** Passport database fields values */
        attributes: PassportAttributes

        /** Connection id */
        connection_id: string // eslint-disable-line camelcase

        display_name?: DisplayName // eslint-disable-line camelcase

        /** Error message (OK if all right) */
        error: string

        /** The remaining validity period of the cookie in seconds. */
        expires_in?: number // eslint-disable-line camelcase

        /** True if user has password hints */
        have_hint?: boolean // eslint-disable-line camelcase

        /** True if user has password */
        have_password?: boolean // eslint-disable-line camelcase

        karma?: Karma

        karma_status?: RawKarma // eslint-disable-line camelcase

        locations: LocationsMap

        /** User login */
        login?: string

        /** @deprecated User login */
        regname?: string

        status: AuthorizationStatus

        /**
         * Session type
         *
         * See:
         * https://nda.ya.ru/3UNLVC
         */
        ttl?: string

        uid?: UserId

        /** User-ticket of user in TVM */
        user_ticket?: string // eslint-disable-line camelcase
    }

    /**
     * Passport user data
     *
     * More information:
     * https://doc.yandex-team.ru/blackbox/reference/method-sessionid-response-json.html
     */
    export interface PassportUser {

        /** User age */
        age?: number

        avatar?: Avatar

        /** User country presented in the form of top level domain (e.g. "ru") */
        country?: string

        /** User display name */
        displayName?: string

        /** User name */
        fio?: string

        /** Error message (OK if all right) */
        error?: string

        /** True if user has password hints */
        haveHint?: boolean

        /** True if user has password */
        havePassword?: boolean

        /**
         * The normalized indicator that the user is a spammer:
         * 0 - not spammer, 100 - sure a spammer.
         */
        karma?: number

        /** User language (e.g. "ru") */
        lang?: string

        /** User login */
        login?: string

        /** User sex (e.g. "f" or "m") */
        sex?: string

        /**
         * String representation of authorization status:
         * VALID - account exists and active
         * NEED_RESET – need to update session cookies
         * DISABLED – account blocked
         * INVALID – account doesn't exist or need to change password
         * INVALID_PARAMS – invalid request params
         * REQUEST_ERROR – error during request
         *
         * See:
         * https://nda.ya.ru/3UNKcC
         */
        status: string

        /**
         * Session type
         *
         * See:
         * https://nda.ya.ru/3UNLVC
         */
        ttl?: string

        /** User ID */
        uid?: string

        /** User-ticket of user in TVM */
        userTicket?: string

        raw: PassportSessionidResponse
    }

    interface AttributesMap {
        [key: string]: string
    }

    export interface ExpressBlackboxOptions extends GotOptions<string | null> {

        /**
         * Email, which is checked on an accessory to the user.
         *
         * Required if value of emails options is "testone"
         */
        addrtotest?: string

        /**
         * Get user login aliases
         *
         * Could be list of numbers separated by commas from this table:
         * https://wiki.yandex-team.ru/passport/dbmoving/#tipyaliasov
         */
        aliases?: 'getsocial' | 'getphonenumber' | 'all' | string

        /** Passport API host */
        api: string

        /**
         * Request user attributes into map
         *
         * Full list:
         * https://wiki.yandex-team.ru/passport/dbmoving#tipyatributov.
         *
         * By default:
         * {
         *   login: '1008',
         *   fio: '1007',
         *   sex: '29',
         *   email: '14',
         *   country: '31',
         *   lang: '34',
         *   avatar: '98'
         * }
         */
        attributes?: AttributesMap

        /**
         * Get additional authorization info,
         * authorization cookie creation date for example
         */
        authid?: 'yes'

        /** Get user emails */
        emails?: 'getall' | 'getyandex' | 'getdefault' | 'testone'

        /** Request format (JSON or XML)  */
        format?: 'json' | 'xml'

        /** Get user real addresses */
        getlocation?: 'postal'

        /** Get user info even if cookie was expired or account was blocked  */
        full_info?: 'yes' // eslint-disable-line camelcase

        getServiceTicket?: (req: Request, res: Response) => Promise<string> | string | undefined

        /** Oauth token, if true it will be extracted from Authorization header */
        oauth?: string | boolean

        /** Number of request addtional attempts */
        retries?: number

        /** Request timeout */
        timeout?: number
}

    /**
     * Authorizes users and places Yandex user data into req.blackbox
     *
     * Source code:
     * https://github.yandex-team.ru/toolbox/express-blackbox
     */
    function expressBlackbox(options?: ExpressBlackboxOptions): RequestHandler;

    export default expressBlackbox;
}
