import { cn } from '@bem-react/classname';
import React, { Component, Fragment } from 'react';
import BansInfo from 'client/components/bansInfo';
import BanForm from 'client/components/banForm';
import UnbanForm from 'client/components/unbanForm';
import StatisticsInput from 'client/components/statisticsInput';
import { IAssociationsInfo, IBanError, IBanInfo, IGlobalUserInfo } from 'client/store/bans/types';
import ERROR_CODES from 'common/types/errorCodes';

import { toastr } from 'react-redux-toastr';
import { Spin } from 'lego-on-react';

import IExam from 'common/types/exam';

import 'lego-on-react/src/components/spin/spin.css';

import './ban.css';
import Dictionary from 'common/types/dictionary';

const b = cn('Ban');

interface IBanProps {
    exams: IExam[],
    examsById: Dictionary<string>,
    login: string,
    hasFocusLogin: boolean,
    isValidLogin: boolean,
    isPending: boolean,
    requestError: IBanError | null,
    associationsInfo: IAssociationsInfo,
    typeLoginAction(inputText: string): void,
    focusLogin(): void,
    blurLogin(): void,
    getBanInfo(login: string): void
}

class Ban extends Component<IBanProps> {
    private readonly showError = (error: IBanError) => {
        const { internalCode } = error;
        const errorHandler = ERROR_CODES[internalCode] || ERROR_CODES.default;

        toastr.error(errorHandler.text, '');
    };

    private readonly getBanInfo = () => {
        const { login, getBanInfo } = this.props;

        if (login) {
            return getBanInfo(login);
        }
    };

    private readonly renderBanInfo = () => {
        const { exams, associationsInfo: { logins, globalUserInfo } } = this.props;
        const { actualLogin, isBanned } = globalUserInfo;

        if (!logins.length && !Object.keys(globalUserInfo).length) {
            return (
                <div className={b('NoData')}>Пользователь не забанен, связанных логинов нет</div>
            );
        }

        return (
            <div className={b('BanInfo')}>
                <div className={b('LoginsInfo')}>
                    <div className={b('Logins')}>
                        <span className={b('LoginsTitle')}>Связанные логины: </span>
                        { logins.join(', ') }
                    </div>
                    <div className={b('ActualLogin')}>
                        <span className={b('LoginsTitle')}>Актуальный логин: </span>{ actualLogin }
                    </div>
                </div>
                <BansInfo isSuperbanned={isBanned} exams={exams} bansInfo={globalUserInfo.bans} />
            </div>
        );
    };

    private readonly getBannedExamsIds = (globalUserInfo: IGlobalUserInfo) => {
        const { bans = [] } = globalUserInfo;
        const now = Date.now();

        const bannedExams = bans.reduce((acc: Dictionary<boolean>, banData: IBanInfo) => {
            const { trialTemplateId, action, isLast, expiredDate } = banData;
            const isNotExpired = Date.parse(expiredDate) >= now || expiredDate === null;

            if (action === 'ban' && isLast && isNotExpired) {
                acc[trialTemplateId] = true;
            }

            return acc;
        }, {});

        return Object.keys(bannedExams);
    };

    /* eslint-disable-next-line complexity */
    private readonly renderAssociations = () => {
        const { associationsInfo, examsById } = this.props;

        const globalUserInfo = associationsInfo && associationsInfo.globalUserInfo;
        const isSuperbanned = globalUserInfo && globalUserInfo.isBanned;

        const bannedExamsIds = this.getBannedExamsIds(globalUserInfo);
        const isBanExist = bannedExamsIds.length !== 0;

        return (
            <Fragment>
                { associationsInfo && this.renderBanInfo() }
                { associationsInfo && !isSuperbanned &&
                <BanForm
                    globalUserInfo={associationsInfo!.globalUserInfo}
                    examsById={examsById}
                    bannedExamsIds={bannedExamsIds}
                    />
                }
                { isBanExist &&
                    <UnbanForm
                        globalUserInfo={associationsInfo!.globalUserInfo}
                        examsById={examsById}
                        bannedExamsIds={bannedExamsIds}
                        />
                }
            </Fragment>
        );
    };

    public componentDidUpdate() {
        const { requestError } = this.props;

        if (requestError) {
            this.showError(requestError);
        }
    }

    public render() {
        const {
            login,
            hasFocusLogin,
            isValidLogin,
            isPending,
            typeLoginAction,
            focusLogin,
            blurLogin,
            associationsInfo
        } = this.props;

        return (
            <div className={b()}>
                <div className={b('Title')}>Бан пользователей</div>
                <div className={b('Input')}>
                    <StatisticsInput
                        tip="Логин пользователя"
                        buttonText="Получить информацию"
                        inputText={login}
                        hasFocus={hasFocusLogin}
                        placeholder="логин"
                        errorMessage={isValidLogin ? undefined : 'Введите логин'}
                        typeAction={typeLoginAction}
                        focus={focusLogin}
                        blur={blurLogin}
                        onSubmit={this.getBanInfo}
                        disabled={isPending || !isValidLogin}
                        />
                </div>
                <div className={b('Loading')}>
                    <Spin size="m" progress={isPending} />
                </div>
                { associationsInfo && this.renderAssociations() }
            </div>
        );
    }
}

export default Ban;
