import without from 'lodash/without';

import { cn } from '@bem-react/classname';

import React, { Component } from 'react';
import { Button, CheckBox, Select, Spin, TextArea } from 'lego-on-react';
import ConfirmationModal from 'client/components/confirmationModal';
import Dictionary from 'common/types/dictionary';
import { toastr } from 'react-redux-toastr';

import './banForm.css';

import 'lego-on-react/src/components/button/button.css';
import 'lego-on-react/src/components/checkbox/checkbox.css';
import 'lego-on-react/src/components/select/select.css';
import 'lego-on-react/src/components/spin/spin.css';
import 'lego-on-react/src/components/textarea/textarea.css';
import { IBanPayload } from 'client/store/banForm/types';
import { IGlobalUserInfo } from 'client/store/bans/types';
import { IDetailsError, getDetailedError } from 'common/utils/error';

const b = cn('BanForm');

interface BanFormProps {
    examsById: Dictionary<string>,
    globalUserInfo: IGlobalUserInfo,
    bannedExamsIds: string[],
    examsForBan: string[],
    logins: string,
    hasFocusLogins: boolean,
    isValidLogins: boolean,
    hasLogins: boolean,
    reason: string,
    hasFocusReason: boolean,
    isSuperban: boolean,
    isPending: boolean,
    requestError: IDetailsError | null,
    isBanSuccess: boolean,
    userLogin: string,
    isVisibleConfirmationModal: boolean,
    submitErrors: string[] | null,
    chooseExams(exams: string[]): void,
    typeLogins(inputText: string): void,
    focusLogins(): void,
    blurLogins(): void,
    typeReason(inputText: string): void,
    focusReason(): void,
    blurReason(): void,
    toggleSuperban(): void,
    pressBan(): void,
    ban(banData: IBanPayload): void,
    decline(): void,
    getBanInfo(login: string): void
}

class BanForm extends Component<BanFormProps> {
    private readonly showError = (error: IDetailsError) => {
        const { message, details } = getDetailedError(error);

        toastr.error(message, details);
    };

    private readonly onBanClick = () => {
        const { pressBan } = this.props;

        pressBan();
    };

    private readonly confirmBan = () => {
        const { logins, examsForBan, reason, isSuperban, ban, examsById } = this.props;
        let examIds = examsForBan;

        if (isSuperban) {
            examIds = Object.keys(examsById);
        }

        return ban({
            logins,
            examsForBan: examIds,
            reason,
            isSuperban
        });
    };

    private readonly getAvailableBans = () => {
        const { examsById, bannedExamsIds } = this.props;

        const examIds = Object.keys(examsById);

        return without(examIds, ...bannedExamsIds);
    };

    private readonly showFormValidationError = () => {
        const { submitErrors } = this.props;
        const error = submitErrors!.join('; ');

        toastr.error('Проверьте поля формы', error);
    };

    public componentDidMount() {
        const { userLogin, typeLogins } = this.props;

        if (userLogin) {
            typeLogins(`${userLogin}, `);
        }
    }

    public componentDidUpdate() {
        const { requestError, isBanSuccess, getBanInfo, userLogin, submitErrors } = this.props;

        if (requestError) {
            this.showError(requestError);
        }

        if (isBanSuccess) {
            toastr.success('Пользователь забанен', '');

            if (userLogin) {
                getBanInfo(userLogin);
            }
        }

        if (submitErrors) {
            this.showFormValidationError();
        }
    }

    public render() {
        const {
            examsForBan,
            examsById,
            logins,
            reason,
            isSuperban,
            toggleSuperban,
            typeLogins,
            focusLogins,
            blurLogins,
            typeReason,
            focusReason,
            blurReason,
            chooseExams,
            isPending,
            isVisibleConfirmationModal,
            decline
        } = this.props;

        const availableForBanExams = this.getAvailableBans();

        return (
            <div className={b()}>
                <form>
                    <div className={b('Title')}>Забанить</div>
                    <label htmlFor="logins" className={b('Label')}>
                        Логины для связывания и бана через запятую
                        <span className={b('RequiredMark')}>*</span>
                    </label>
                    <TextArea
                        cls={b('Field')}
                        theme="normal"
                        placeholder="Логины для бана"
                        size="m"
                        name="logins"
                        rows={2}
                        text={logins}
                        onChange={typeLogins}
                        onFocus={focusLogins}
                        onBlur={blurLogins}
                        disabled={isPending}
                        />
                    <Select
                        theme="normal"
                        cls={b('Field')}
                        size="m"
                        type="check"
                        text="vary"
                        width="max"
                        name="selectBan"
                        placeholder="Бан в тестах"
                        val={examsForBan}
                        disabled={!availableForBanExams.length || isPending || isSuperban}
                        onChange={chooseExams}
                        >
                        {
                            availableForBanExams.map((id: string) => {
                                const slug = examsById[id];

                                return (
                                    <Select.Item key={id} val={id}>
                                        {slug}
                                    </Select.Item>
                                );
                            })}
                    </Select>
                    <label htmlFor="reason" className={b('Label')}>
                        Причина бана<span className={b('RequiredMark')}>*</span>
                    </label>
                    <TextArea
                        cls={b('Field')}
                        theme="normal"
                        placeholder="Причина бана"
                        size="m"
                        name="reason"
                        rows={2}
                        text={reason}
                        onChange={typeReason}
                        onFocus={focusReason}
                        onBlur={blurReason}
                        disabled={isPending}
                        />
                    <CheckBox
                        cls={b('Field')}
                        theme="normal"
                        size="m"
                        text="Супербан"
                        checked={isSuperban}
                        onChange={toggleSuperban}
                        disabled={isPending}
                        />
                    <div className={b('BanActions')}>
                        <Button
                            cls={b('BanButton')}
                            theme="action"
                            size="m"
                            text="Забанить"
                            onClick={this.onBanClick}
                            disabled={isPending}
                            />
                        <div className={b('Loading')}>
                            <Spin size="m" progress={isPending} />
                        </div>
                    </div>
                </form>
                <ConfirmationModal
                    confirmationText="Вы уверены?"
                    confirmText="Да"
                    declineText="Нет"
                    onConfirm={this.confirmBan}
                    onDecline={decline}
                    visible={isVisibleConfirmationModal}
                    />
            </div>
        );
    }
}

export default BanForm;
