import { cn } from '@bem-react/classname';
import React, { Component } from 'react';
import { Button, Icon, TextInput, TextArea } from 'lego-on-react';
import { toastr } from 'react-redux-toastr';

import 'lego-on-react/src/components/button/button.css';
import 'lego-on-react/src/components/textinput/textinput.css';
import 'lego-on-react/src/components/textarea/textarea.css';
import './editExamForm.css';
import '../icon/icon.css';

import IExamSettings from '../../../common/types/examSettings';
import ERROR_CODES from '../../../common/types/errorCodes';
import { ISaveFormError } from '../../store/editExamForm/types';

const b = cn('EditExamForm');

interface IEditExamFormProps {
    isReadonly: boolean,
    slug: string,
    timeLimit: number,
    title: string,
    description: string,
    rules: string,
    seoDescription: string,
    ogDescription: string,
    isEditingTitle: boolean,
    isSavePending: boolean,
    isSaveSuccess: boolean,
    saveError: ISaveFormError | null,
    changeTimeLimitAction(timeLimit: number): void,
    changeRulesAction(rules: string): void,
    changeDescriptionAction(description: string): void,
    changeSeoDescriptionAction(seoDescription: string): void,
    changeOgDescriptionAction(ogDescription: string): void,
    changeTitleAction(title: string): void,
    switchTitleEditAction(isEditingTitle: boolean): void,
    saveSettingsAction(exam: IExamSettings): void
}

const msInMinute = 60 * 1000;

class EditExamForm extends Component<IEditExamFormProps> {
    private onTimeLimitChange = (timeLimit: string) => {
        this.props.changeTimeLimitAction(Number(timeLimit) * msInMinute);
    };

    private onTitleEditClick = () => {
        this.props.switchTitleEditAction(true);
    };

    private onTitleSaveClick = () => {
        this.props.switchTitleEditAction(false);
    };

    private onSaveSettingsClick = () => {
        const {
            title,
            slug,
            timeLimit,
            description,
            rules,
            seoDescription,
            ogDescription,
            saveSettingsAction
        } = this.props;

        saveSettingsAction({
            slug,
            title,
            timeLimit,
            description,
            rules,
            seoDescription,
            ogDescription
        });
    };

    private getTitle = () => {
        const {
            title,
            isEditingTitle,
            changeTitleAction,
            isSavePending,
            isReadonly
        } = this.props;

        const disabled = isReadonly || isSavePending;

        if (!isEditingTitle) {
            return (
                <div className={b('Title')}>
                    <span className={b('TitleText')}>{title}</span>
                    {
                        !disabled && <Button
                            theme="normal"
                            size="m"
                            onClick={this.onTitleEditClick}
                            pin="round-round"
                            icon={<Icon type="edit" />}
                            disabled={disabled}
                            />
                    }
                </div>
            );
        }

        return (
            <div className={b('Title')}>
                <TextInput
                    cls={b('TitleInput')}
                    theme="normal"
                    size="m"
                    pin="round-round"
                    name="title"
                    text={title}
                    onChange={changeTitleAction}
                    disabled={disabled}
                    />
                <Button
                    theme="normal"
                    size="m"
                    pin="round-round"
                    onClick={this.onTitleSaveClick}
                    icon={<Icon type="accept" />}
                    disabled={disabled}
                    />
            </div>
        );
    };

    private showError = (error: ISaveFormError) => {
        const { internalCode } = error;
        const errorHandler = ERROR_CODES[internalCode] || ERROR_CODES.default;

        toastr.error(errorHandler.text, '');
    };

    public componentDidUpdate() {
        const { isSaveSuccess, saveError } = this.props;

        if (saveError) {
            this.showError(saveError);
        }

        if (isSaveSuccess) {
            toastr.success('Настройки сохранены', '');
        }
    }

    public render() {
        const {
            isReadonly,
            timeLimit,
            description,
            rules,
            ogDescription,
            seoDescription,
            isSavePending,
            changeRulesAction,
            changeDescriptionAction,
            changeSeoDescriptionAction,
            changeOgDescriptionAction
        } = this.props;

        const timeInMinutes = timeLimit / msInMinute;
        const disabled = isReadonly || isSavePending;

        return (
            <form className={b()}>
                {this.getTitle()}
                <div className={b('TimeLimit')}>
                    <label>
                        Время на прохождение теста
                        <TextInput
                            cls={b('TimeInput')}
                            theme="normal"
                            type="number"
                            size="s"
                            controlAttrs={{ min: 0 }}
                            pin="round-round"
                            text={timeInMinutes}
                            onChange={this.onTimeLimitChange}
                            disabled={disabled}
                            />
                        минут
                    </label>
                </div>
                <div className={b('TextField')}>
                    <label>
                        <div className={b('FieldTitle')}>Описание теста (html)</div>
                        <TextArea
                            cls={b('TextArea')}
                            theme="normal"
                            size="m"
                            text={description}
                            name="description"
                            rows={15}
                            onChange={changeDescriptionAction}
                            disabled={disabled}
                            />
                    </label>
                </div>
                <div className={b('TextField')}>
                    <label>
                        <div className={b('FieldTitle')}>Правила тестирования (html)</div>
                        <TextArea
                            cls={b('TextArea')}
                            theme="normal"
                            size="m"
                            text={rules}
                            name="rules"
                            rows={15}
                            onChange={changeRulesAction}
                            disabled={disabled}
                            />
                    </label>
                </div>
                <div className={b('TextField')}>
                    <label>
                        <div className={b('FieldTitle')}>SEO описание (текст)</div>
                        <TextArea
                            cls={b('TextArea')}
                            theme="normal"
                            size="m"
                            text={seoDescription}
                            name="seoDescription"
                            rows={4}
                            onChange={changeSeoDescriptionAction}
                            disabled={disabled}
                            />
                    </label>
                </div>
                <div className={b('TextField')}>
                    <label>
                        <div className={b('FieldTitle')}>OG описание (текст)</div>
                        <TextArea
                            cls={b('TextArea')}
                            theme="normal"
                            size="m"
                            text={ogDescription}
                            name="ogDescription"
                            rows={4}
                            onChange={changeOgDescriptionAction}
                            disabled={disabled}
                            />
                    </label>
                </div>
                <div className={b('Save')}>
                    <Button
                        theme="action"
                        size="m"
                        text="Сохранить"
                        onClick={this.onSaveSettingsClick}
                        disabled={disabled}
                        />
                </div>
            </form>
        );
    }
}

export default EditExamForm;
