import { cn } from '@bem-react/classname';
import { classnames } from '@bem-react/classnames';

import React, { Component, createRef } from 'react';
import { Icon, Tumbler } from 'lego-on-react';
import ContentEditable, { ContentEditableEvent } from 'react-contenteditable';
import ExamElementRestore from 'client/components/examElementRestore';

import 'lego-on-react/src/components/icon/icon.css';
import 'lego-on-react/src/components/tumbler/tumbler.css';

import './examAnswer.css';
import 'client/components/tumbler/tumbler.css';

import { escapeHtml } from 'client/store/examData/helpers';

import { TEditAnswer } from 'common/types/editTestData';
import {
    IChangeAnswerTextPayload,
    IEditAnswerCorrectPayload,
    IToggleRemoveAnswerPayload
} from 'client/store/examData/types';

const b = cn('ExamAnswer');

interface ExamAnswerProps {
    sectionCode: string,
    categoryId: number,
    questionId: number,
    answer: TEditAnswer,
    isReadonly: boolean,
    ref?: any,
    editAnswerCorrect(data: IEditAnswerCorrectPayload): void,
    toggleRemoveAnswer(data: IToggleRemoveAnswerPayload): void,
    changeAnswerText(data: IChangeAnswerTextPayload): void
}

interface ExamAnswerState {
    tempText: string
}

class ExamAnswer extends Component<ExamAnswerProps, ExamAnswerState> {
    constructor(props: ExamAnswerProps) {
        super(props);

        this.state = { tempText: props.answer.text };
    }

    private readonly editableRef = createRef<HTMLElement>();

    private readonly changeCorrect = () => {
        const {
            answer: { id, correct },
            sectionCode,
            categoryId,
            questionId,
            editAnswerCorrect
        } = this.props;

        const newCorrect = correct === 0 ? 1 : 0;

        editAnswerCorrect({
            sectionCode,
            categoryId,
            questionId,
            answerId: id,
            correct: newCorrect
        });
    };

    private readonly remove = () => {
        const {
            sectionCode,
            categoryId,
            questionId,
            answer: { id },
            toggleRemoveAnswer
        } = this.props;

        toggleRemoveAnswer({
            sectionCode,
            categoryId,
            questionId,
            answerId: id,
            isRemoved: true
        });
    };

    private readonly restore = () => {
        const {
            sectionCode,
            categoryId,
            questionId,
            answer: { id },
            toggleRemoveAnswer
        } = this.props;

        toggleRemoveAnswer({
            sectionCode,
            categoryId,
            questionId,
            answerId: id,
            isRemoved: false
        });
    };

    private readonly onChangeAnswerText = () => {
        const {
            sectionCode,
            categoryId,
            questionId,
            answer: { id },
            changeAnswerText
        } = this.props;

        const { tempText } = this.state;
        const escapedTempText = escapeHtml(tempText);

        this.setState({ tempText: escapedTempText });

        changeAnswerText({
            sectionCode,
            categoryId,
            questionId,
            answerId: id,
            text: escapedTempText
        });
    };

    private readonly onChangeTempText = (event: ContentEditableEvent) => {
        this.setState({ tempText: event.target.value });
    };

    public focusEditable() {
        if (this.editableRef.current) {
            this.editableRef.current.focus();
        }
    }

    public render() {
        const {
            isReadonly,
            answer: { correct, isRemoved, isNew, isChanged }
        } = this.props;
        const { tempText } = this.state;
        const isCorrect = Boolean(correct);

        if (isRemoved) {
            return (
                <div className={b({ removed: true })}>
                    <ExamElementRestore
                        text="Восстановить ответ"
                        restoreAction={this.restore}
                        />
                </div>
            );
        }

        const hasChanges = isNew || isChanged;

        return (
            <div className={b({ correct: isCorrect, unsaved: hasChanges })}>
                <div
                    className={b('Remove', { hidden: isReadonly })}
                    onClick={this.remove}
                    >
                    <Icon cls={b('RemoveIcon')} type="gray-cross" />
                </div>
                <div className={b('Content')}>
                    <div className={b('TextWrapper')}>
                        {
                            hasChanges &&
                            <div className={b('Unsaved')}>
                                <span>Неопубликованная версия</span>
                            </div>
                        }
                        <ContentEditable
                            innerRef={this.editableRef}
                            className={b('Text')}
                            html={tempText}
                            disabled={isReadonly}
                            onChange={this.onChangeTempText}
                            onBlur={this.onChangeAnswerText}
                            />
                    </div>
                    <div className={b('Correct')}>
                        <span className={b('CorrectLabel')}>Правильный?</span>
                        <Tumbler
                            checked={isCorrect}
                            size="xs"
                            theme="normal"
                            cls={classnames(b('Tumbler'), cn('tumbler_type_round')())}
                            disabled={isReadonly}
                            onChange={this.changeCorrect}
                            />
                    </div>
                </div>
            </div>
        );
    }
}

export default ExamAnswer;
