import { cn } from '@bem-react/classname';
import React, { Component, createRef, Fragment } from 'react';
import { toastr } from 'react-redux-toastr';

import ExamLoader from 'client/components/examLoader';
import ExamSections from 'client/components/examSections';
import ExamQuestionsModal from 'client/components/examQuestionsModal';
import ConfirmationModal from 'client/components/confirmationModal';

import { Button, Spin } from 'lego-on-react';

import 'lego-on-react/src/components/spin/spin.css';

import './examData.css';

import RequestStatus from 'common/types/requestStatus';
import { getDetailedError, IDetailsError } from 'common/utils/error';
import ITestData from 'common/types/testData';
import IEditTestData from 'common/types/editTestData';
import DraftStatus from 'common/types/draftStatus';
import { validateSections } from 'client/store/examData/helpers';
import { Roles } from 'common/types/adminData';

interface IExamDataProps {
    draftStatus: DraftStatus | null,
    loadStatus: RequestStatus | null,
    startEditStatus: RequestStatus | null,
    publishStatus: RequestStatus | null,
    requestError: IDetailsError | null,
    exam: ITestData | null,
    editData: IEditTestData | null,
    examId: number,
    examSlug: string,
    openedCategoryId: number | null,
    openedSectionCode: string | null,
    isVisiblePublishModal: boolean,
    isVisibleDeleteDraftModal: boolean,
    isVisibleCancelChangesModal: boolean,
    cancelChangesStatus: RequestStatus | null,
    userRoles: Roles[],
    isEditMode: boolean,
    saveDraftStatus: RequestStatus | null,
    sendDraftForModerationStatus: RequestStatus | null,
    deleteDraftStatus: RequestStatus | null,
    isLocked: boolean,
    loadFile(examSlug: string, formData: FormData): void,
    downloadJSON(examSlug: string): void,
    publishExam(examSlug: string, editData: IEditTestData, examId: number): void,
    declinePublish(): void,
    showPublishModal(): void,
    deleteDraft(examSlug: string): void,
    showDeleteDraftModal(): void,
    declineDeleteDraft(): void,
    startEdit(examSlug: string): void,
    saveDraft(examSlug: string, examId: number, editData: IEditTestData): void,
    sendDraftForModeration(examSlug: string, examId: number, editData: IEditTestData): void,
    cancelChanges(examSlug: string): void,
    showCancelChangesModal(): void,
    declineCancelChanges(): void
}

const b = cn('ExamData');

class ExamData extends Component<IExamDataProps> {
    private readonly questionsPopupRef = createRef<HTMLDivElement>();

    private hasEditAccess = () => {
        const { userRoles } = this.props;

        return userRoles.some(role => role === Roles.admin)
            || userRoles.some(role => role === Roles.editor);
    };

    private hasPublishAccess = () => {
        const { userRoles } = this.props;

        return userRoles.some(role => role === Roles.admin);
    };

    private readonly isReadonly = () => {
        const { isEditMode } = this.props;

        return !this.hasEditAccess() || !isEditMode;
    };

    private readonly isQuestionsModalOpened = () => {
        const { editData, openedCategoryId, openedSectionCode } = this.props;

        return editData && openedCategoryId && openedSectionCode;
    };

    private showError = (error: IDetailsError) => {
        const { message, details } = getDetailedError(error);

        toastr.error(message, details);
    };

    private readonly showValidationErrors = (validationErrors: string[]) => {
        const details = validationErrors.join('</br>');

        toastr.error('Невалидные данные в черновике:', '', {
            component: <div dangerouslySetInnerHTML={{ __html: details }} />
        });
    };

    // eslint-disable-next-line complexity,max-statements
    public componentDidUpdate() {
        const {
            requestError,
            publishStatus,
            saveDraftStatus,
            sendDraftForModerationStatus,
            deleteDraftStatus
        } = this.props;

        if (requestError) {
            this.showError(requestError);
        }

        if (this.isQuestionsModalOpened()) {
            document.body.style.overflowY = 'hidden';
        } else {
            document.body.style.overflowY = 'visible';
        }

        if (publishStatus === RequestStatus.success) {
            toastr.success('Данные обновлены', '');
        }

        if (saveDraftStatus === RequestStatus.success) {
            toastr.success('Черновик сохранен', '');
        }

        if (sendDraftForModerationStatus === RequestStatus.success) {
            toastr.success('Черновик отправлен на модерацию', '');
        }

        if (deleteDraftStatus === RequestStatus.success) {
            toastr.success('Черновик удален', '');
        }
    }

    private getCounters = () => {
        const { editData } = this.props;

        if (!editData) {
            return <div className={b('Counters')} />;
        }

        const sectionsKeys = Object.keys(editData);
        const questionsCount = sectionsKeys.reduce((count, sectionKey) => {
            const categoriesIds = Object.keys(editData[sectionKey].categories);

            categoriesIds.forEach(categoryId => {
                count += Object.keys(editData[sectionKey]
                    .categories[Number(categoryId)].questions).length;
            });

            return count;
        }, 0);

        return (
            <div className={b('Counters')}>
                <div className={b('Counter')}>Количество cекций:
                    <span className={b('Count')}> {sectionsKeys.length}</span>
                </div>
                <div className={b('Counter')}>Количество вопросов:
                    <span className={b('Count')}> {questionsCount}</span>
                </div>
            </div>
        );
    };

    private getPublishButton = (hasPendingRequest: boolean) => {
        const { showPublishModal } = this.props;

        return (
            <Fragment>
                {
                    this.hasPublishAccess() ?
                        <Button
                            theme="action"
                            size="m"
                            cls={b('PublishButton')}
                            text="Опубликовать"
                            disabled={hasPendingRequest}
                            onClick={showPublishModal}
                            />
                        :
                        <Button
                            theme="action"
                            size="m"
                            cls={b('PublishButton')}
                            text="Отправить на модерацию"
                            disabled={hasPendingRequest}
                            onClick={this.onSendDraftForModerationClick}
                            />
                }
            </Fragment>
        );
    };

    private getButtons = (hasPendingRequest: boolean) => {
        const {
            isEditMode,
            showDeleteDraftModal,
            showCancelChangesModal,
            draftStatus
        } = this.props;

        if (!isEditMode) {
            return;
        }

        const isDraftExist = draftStatus === DraftStatus.saved ||
            draftStatus === DraftStatus.onModeration;

        return (
            <div className={b('Save')}>
                <div className={b('ButtonsContainer')}>
                    <div className={b('RemoveButtons')}>
                        { isEditMode && <Button
                            theme="normal"
                            size="m"
                            cls={b('RemoveButton')}
                            text="Выйти из редактирования"
                            disabled={hasPendingRequest}
                            onClick={showCancelChangesModal}
                            />
                        }
                        { isDraftExist && <Button
                            theme="normal"
                            size="m"
                            cls={b('RemoveButton')}
                            text="Удалить черновик"
                            disabled={hasPendingRequest}
                            onClick={showDeleteDraftModal}
                            />
                        }
                    </div>
                    <div className={b('SaveButtons')}>
                        <Button
                            theme="normal"
                            size="m"
                            cls={b('SaveDraftButton')}
                            text="Сохранить черновик"
                            disabled={hasPendingRequest}
                            onClick={this.onSaveDraftClick}
                            />
                        { this.getPublishButton(hasPendingRequest) }
                    </div>
                </div>
            </div>
        );
    };

    private validateData = () => {
        const { editData, exam } = this.props;

        if (!editData || !exam) {
            return false;
        }

        const validationErrors = validateSections(editData);

        if (validationErrors.length > 0) {
            this.showValidationErrors(validationErrors);

            return false;
        }

        return true;
    };

    private onPublishExamClick = () => {
        const { examSlug, examId, editData, publishExam } = this.props;
        const isValid = this.validateData();

        if (isValid) {
            publishExam(examSlug, editData!, examId);
        }
    };

    private onDeleteDraftClick = () => {
        const { examSlug, deleteDraft } = this.props;

        deleteDraft(examSlug);
    };

    private onCancelChangesClick = () => {
        const { examSlug, cancelChanges } = this.props;

        cancelChanges(examSlug);
    };

    private onSaveDraftClick = () => {
        const { examId, editData, examSlug, saveDraft } = this.props;
        const isValid = this.validateData();

        if (isValid) {
            saveDraft(examSlug, examId, editData!);
        }
    };

    private onSendDraftForModerationClick = () => {
        const { examId, editData, examSlug, sendDraftForModeration } = this.props;
        const isValid = this.validateData();

        if (isValid) {
            sendDraftForModeration(examSlug, examId, editData!);
        }
    };

    /* eslint-disable-next-line complexity */
    public render() {
        const {
            loadFile,
            downloadJSON,
            editData,
            examSlug,
            loadStatus,
            openedCategoryId,
            openedSectionCode,
            isVisiblePublishModal,
            isVisibleDeleteDraftModal,
            isVisibleCancelChangesModal,
            declineCancelChanges,
            cancelChangesStatus,
            declinePublish,
            declineDeleteDraft,
            publishStatus,
            startEdit,
            startEditStatus,
            saveDraftStatus,
            deleteDraftStatus,
            isEditMode,
            isLocked,
            draftStatus,
            sendDraftForModerationStatus
        } = this.props;

        const hasPendingRequest = [
            startEditStatus,
            loadStatus,
            cancelChangesStatus,
            deleteDraftStatus,
            saveDraftStatus,
            sendDraftForModerationStatus,
            publishStatus
        ].some(status => status === RequestStatus.pending);

        return (
            <div className={b()}>
                <div className={b('Header')}>
                    { this.getCounters() }
                    { !isEditMode && <ExamLoader
                        loadFile={loadFile}
                        downloadJSON={downloadJSON}
                        startEdit={startEdit}
                        hasEditAccess={this.hasEditAccess()}
                        hasPublishAccess={this.hasPublishAccess()}
                        examSlug={examSlug}
                        disabled={hasPendingRequest}
                        isLocked={isLocked}
                        draftStatus={draftStatus}
                        />
                    }
                    { hasPendingRequest &&
                        <div className={b('Loading')}>
                            <Spin size="l" progress />
                        </div>
                    }
                </div>
                { editData && <ExamSections
                    editData={editData}
                    isReadonly={this.isReadonly()}
                    questionsPopupRef={this.questionsPopupRef}
                    />

                }
                { this.getButtons(hasPendingRequest) }
                {
                    this.isQuestionsModalOpened() &&
                    <ExamQuestionsModal
                        editData={editData!}
                        categoryId={openedCategoryId!}
                        sectionCode={openedSectionCode!}
                        anchorRef={this.questionsPopupRef.current}
                        isReadonly={this.isReadonly()}
                        />
                }
                <ConfirmationModal
                    confirmationText="Точно опубликовать вопросы?"
                    confirmText="Да"
                    declineText="Нет"
                    onConfirm={this.onPublishExamClick}
                    onDecline={declinePublish}
                    visible={isVisiblePublishModal}
                    />
                <ConfirmationModal
                    confirmationText="Точно удалить черновик? Восстановить
                        изменения будет невозможно!"
                    confirmText="Да"
                    declineText="Нет"
                    onConfirm={this.onDeleteDraftClick}
                    onDecline={declineDeleteDraft}
                    visible={isVisibleDeleteDraftModal}
                    />
                <ConfirmationModal
                    confirmationText="Точно выйти из режима редактирования? Восстановить
                        изменения будет невозможно!"
                    confirmText="Да"
                    declineText="Нет"
                    onConfirm={this.onCancelChangesClick}
                    onDecline={declineCancelChanges}
                    visible={isVisibleCancelChangesModal}
                    />
            </div>
        );
    }
}

export default ExamData;
