import { cn } from '@bem-react/classname';
import React, { Component } from 'react';
import { Attach, Button } from 'lego-on-react';

import 'lego-on-react/src/components/attach/attach.css';
import './examLoader.css';

import DraftStatus from 'common/types/draftStatus';

const b = cn('ExamLoader');

interface IExamLoaderProps {
    loadFile(examSlug: string, formData: FormData): void,
    downloadJSON(examSlug: string): void,
    startEdit(examSlug: string): void,
    disabled: boolean,
    examSlug: string,
    hasEditAccess: boolean,
    hasPublishAccess: boolean,
    isLocked: boolean,
    draftStatus: DraftStatus | null
}

class ExamLoader extends Component<IExamLoaderProps> {
    private onFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files && e.target.files[0];

        if (!file) {
            return;
        }

        const { examSlug, loadFile } = this.props;

        const formData = new FormData();

        formData.append('file', file);

        loadFile(examSlug, formData);
    };

    private onDownloadJSONClick = () => {
        const { downloadJSON, examSlug } = this.props;

        downloadJSON(examSlug);
    };

    private onCreateDraftClick = () => {
        const { startEdit, examSlug } = this.props;

        startEdit(examSlug);
    };

    private getDraftText = () => {
        const { draftStatus, hasPublishAccess } = this.props;
        const continueText = 'Продолжить редактирование';

        if (draftStatus === DraftStatus.saved) {
            return continueText;
        }

        if (draftStatus === DraftStatus.onModeration) {
            return hasPublishAccess ? 'Открыть черновик на модерации' : continueText;
        }

        return 'Начать редактирование';
    };

    public render() {
        const { disabled, hasEditAccess, isLocked } = this.props;

        return (
            <div className={b()}>
                { hasEditAccess &&
                    <Button
                        theme="normal"
                        size="m"
                        cls={b('Draft')}
                        text={this.getDraftText()}
                        disabled={disabled || isLocked}
                        onClick={this.onCreateDraftClick}
                        />
                }
                <Attach
                    theme="normal"
                    size="m"
                    cls={b('Attach')}
                    // @ts-ignore
                    holder
                    disabled={disabled || isLocked}
                    onChange={this.onFileChange}
                    />
                <Button
                    theme="normal"
                    size="m"
                    cls={b('DownloadJSON')}
                    text="Посмотреть текущую версию"
                    disabled={disabled}
                    onClick={this.onDownloadJSONClick}
                    />
            </div>
        );
    }
}

export default ExamLoader;
