import { cn } from '@bem-react/classname';
import React, { Component, createRef } from 'react';
import { Icon, Select } from 'lego-on-react';
import ExamAnswers from 'client/components/examAnswers';
import ExamElementRestore from 'client/components/examElementRestore';
import ContentEditable, { ContentEditableEvent } from 'react-contenteditable';

import 'lego-on-react/src/components/select/select.css';

import './examQuestion.css';
import { TEditQuestion } from 'common/types/editTestData';
import {
    IChangeQuestionTextPayload,
    IEditQuestionTypePayload,
    IToggleRemoveQuestionPayload
} from 'client/store/examData/types';
import { itemsHasChanges, escapeHtml } from 'client/store/examData/helpers';

const b = cn('ExamQuestion');

interface ExamQuestionProps {
    sectionCode: string,
    categoryId: number,
    question: TEditQuestion,
    isReadonly: boolean,
    editQuestionType(data: IEditQuestionTypePayload): void,
    toggleRemoveQuestion(data: IToggleRemoveQuestionPayload): void,
    changeQuestionText(data: IChangeQuestionTextPayload): void
}

interface IExamQuestionState {
    isExpanded: boolean,
    tempText: string
}

class ExamQuestion extends Component<ExamQuestionProps, IExamQuestionState> {
    constructor(props: ExamQuestionProps) {
        super(props);

        this.state = { isExpanded: false, tempText: props.question.text };
    }

    private readonly editableRef = createRef<HTMLElement>();

    private readonly selectItems = [
        { val: 0, text: 'Один' },
        { val: 1, text: 'Несколько' }
    ];

    private toggleExpand = () => {
        const { isExpanded } = this.state;

        this.setState({ isExpanded: !isExpanded });
    };

    private getToggleAnswersText = () => {
        const { isExpanded } = this.state;

        return isExpanded ? 'Скрыть ответы' : 'Показать ответы';
    };

    private onEditQuestionType = (value: string[]) => {
        const {
            question: { id },
            sectionCode,
            categoryId,
            editQuestionType
        } = this.props;

        editQuestionType({
            sectionCode,
            categoryId,
            questionId: id,
            type: Number(value[0])
        });
    };

    private readonly remove = () => {
        const { sectionCode, categoryId, question: { id }, toggleRemoveQuestion } = this.props;

        toggleRemoveQuestion({
            sectionCode,
            categoryId,
            questionId: id,
            isRemoved: true
        });
    };

    private readonly restore = () => {
        const { sectionCode, categoryId, question: { id }, toggleRemoveQuestion } = this.props;

        toggleRemoveQuestion({
            sectionCode,
            categoryId,
            questionId: id,
            isRemoved: false
        });
    };

    private readonly onChangeTempText = (event: ContentEditableEvent) => {
        this.setState({ tempText: event.target.value });
    };

    private readonly onChangeQuestionText = () => {
        const { sectionCode, categoryId, question: { id }, changeQuestionText } = this.props;
        const { tempText } = this.state;
        const escapedTempText = escapeHtml(tempText);

        this.setState({ tempText: escapedTempText });

        changeQuestionText({
            sectionCode,
            categoryId,
            questionId: id,
            text: escapedTempText
        });
    };

    private readonly hasChanges = () => {
        const { question: { isNew, isChanged, answers } } = this.props;

        return isNew || isChanged || itemsHasChanges(answers);
    };

    public focusEditable() {
        if (this.editableRef.current) {
            this.editableRef.current.focus();
        }
    }

    public render() {
        const {
            sectionCode,
            categoryId,
            question: { type, id, answers, isRemoved },
            isReadonly
        } = this.props;
        const { isExpanded, tempText } = this.state;

        if (isRemoved) {
            return (
                <div className={b({ removed: true })}>
                    <ExamElementRestore
                        text="Восстановить вопрос"
                        restoreAction={this.restore}
                        />
                </div>
            );
        }

        const hasChanges = this.hasChanges();

        return (
            <div className={b({ expanded: isExpanded, unsaved: hasChanges })}>
                <div
                    className={b('Remove', { hidden: isReadonly })}
                    onClick={this.remove}
                    >
                    <Icon cls={b('RemoveIcon')} type="gray-cross" />
                </div>
                <div className={b('Content')}>
                    <div className={b('TextWrapper')}>
                        {
                            hasChanges &&
                            <div className={b('Unsaved')}>
                                <span>Неопубликованная версия</span>
                            </div>
                        }
                        <ContentEditable
                            className={b('Text')}
                            innerRef={this.editableRef}
                            html={tempText}
                            disabled={isReadonly}
                            onChange={this.onChangeTempText}
                            onBlur={this.onChangeQuestionText}
                            />
                    </div>
                    <div className={b('Type')}>
                        <span className={b('Type-Label')}>Верных ответов</span>
                        <Select
                            theme="normal"
                            type="radio"
                            size="xs"
                            text="vary"
                            items={this.selectItems}
                            val={type}
                            width="max"
                            cls={b('TypeSelect')}
                            disabled={isReadonly}
                            onChange={this.onEditQuestionType}
                            />
                    </div>
                    <button className={b('ToggleAnswers')} onClick={this.toggleExpand}>
                        <span>{this.getToggleAnswersText()}</span>
                        <Icon type="triangle" cls={b('ToggleIcon')} />
                    </button>
                    <div className={b('Answers')}>
                        <ExamAnswers
                            sectionCode={sectionCode}
                            categoryId={categoryId}
                            questionId={id}
                            answers={answers}
                            isReadonly={isReadonly}
                            />
                    </div>
                </div>
            </div>
        );
    }
}

export default ExamQuestion;
