import { cn } from '@bem-react/classname';
import React, { Component, createRef, Fragment } from 'react';
import ExamQuestion from 'client/components/examQuestion';
import NumberInput from 'client/components/numberInput';
import { Button } from 'lego-on-react';

import './examQuestions.css';
import 'lego-on-react/src/components/button/button.css';
import 'lego-on-react/src/components/textinput/textinput.css';

import IEditTestData, { TEditSection, IEditQuestions } from 'common/types/editTestData';
import { IAddQuestionPayload, IEditCategoryQuantityPayload } from 'client/store/examData/types';
import { countActiveItems } from 'client/store/examData/helpers';

const b = cn('ExamQuestions');

interface ExamQuestionsProps {
    editData: IEditTestData,
    sectionCode: string,
    categoryId: number,
    isReadonly: boolean,
    editCategoryQuantity(data: IEditCategoryQuantityPayload): void,
    addQuestion(data: IAddQuestionPayload): void
}

class ExamQuestions extends Component<ExamQuestionsProps> {
    private readonly firstNewQuestionRef = createRef<any>();

    private readonly onQuantityChange = (quantity: number) => {
        const { sectionCode, categoryId, editCategoryQuantity, editData } = this.props;
        const questionsInCategory = editData[sectionCode].categories[categoryId].questions;
        const activeQuestionsCount = countActiveItems(questionsInCategory);

        if (quantity > 0 && quantity <= activeQuestionsCount) {
            editCategoryQuantity({
                sectionCode,
                categoryId,
                quantity
            });
        }
    };

    private readonly onAddQuestion = () => {
        const { sectionCode, categoryId, addQuestion } = this.props;

        addQuestion({
            sectionCode,
            categoryId
        });
    };

    private readonly renderNewQuestions = (questions: IEditQuestions) => {
        const { sectionCode, categoryId, isReadonly } = this.props;
        const questionsIds = Object.keys(questions);
        const newQuestionsIds = questionsIds
            .filter((questionId: string) => {
                return questions[Number(questionId)].isNew;
            })
            .reverse();

        if (!newQuestionsIds.length) {
            return null;
        }

        const firstQuestionId = newQuestionsIds.shift();

        return (
            <Fragment>
                {
                    <ExamQuestion
                        key={firstQuestionId}
                        sectionCode={sectionCode}
                        categoryId={categoryId}
                        question={questions[Number(firstQuestionId)]}
                        isReadonly={isReadonly}
                        ref={this.firstNewQuestionRef}
                        />
                }
                {
                    newQuestionsIds.map(questionId => {
                        return (
                            <ExamQuestion
                                key={questionId}
                                sectionCode={sectionCode}
                                categoryId={categoryId}
                                question={questions[Number(questionId)]}
                                isReadonly={isReadonly}
                                />
                        );
                    })
                }
            </Fragment>
        );
    };

    private readonly renderOldQuestions = (questions: IEditQuestions) => {
        const { sectionCode, categoryId, isReadonly } = this.props;
        const questionsIds = Object.keys(questions);
        const oldQuestionsIds = questionsIds.filter((questionId: string) => {
            return !questions[Number(questionId)].isNew;
        });

        return (
            <Fragment>
                {
                    oldQuestionsIds.map(questionId => {
                        return (
                            <ExamQuestion
                                key={questionId}
                                sectionCode={sectionCode}
                                categoryId={categoryId}
                                question={questions[Number(questionId)]}
                                isReadonly={isReadonly}
                                />
                        );
                    })
                }
            </Fragment>
        );
    };

    public componentDidUpdate(prevProps: Readonly<ExamQuestionsProps>): void {
        const { editData, sectionCode, categoryId } = this.props;
        const {
            editData: oldEditData,
            sectionCode: oldSectionCode,
            categoryId: oldCategoryId
        } = prevProps;
        const { questions: oldQuestions } = oldEditData[oldSectionCode].categories[oldCategoryId];
        const { questions: newQuestions } = editData[sectionCode].categories[categoryId];

        const oldLength = Object.keys(oldQuestions).length;
        const currentLength = Object.keys(newQuestions).length;

        if (currentLength > oldLength) {
            this.firstNewQuestionRef.current.getWrappedInstance().focusEditable();
        }
    }

    public render() {
        const { editData, sectionCode, categoryId, isReadonly } = this.props;

        const section = editData[sectionCode] as TEditSection;
        const { title: sectionTitle } = section;
        const category = section.categories[categoryId];
        const { questions, quantity } = category;

        return (
            <div className={b()}>
                <div className={b('Header')}>
                    <div className={b('Title')}>
                        <span className={b('SectionTitle')}>{sectionTitle}. </span>
                        <span className={b('Category')}>Категория:
                            <span className={b('CategoryId')}> {categoryId}</span>
                        </span>
                    </div>
                    <div className={b('Quantity')}>
                        <NumberInput
                            min={1}
                            max={Object.keys(questions).length}
                            value={quantity}
                            typeAction={this.onQuantityChange}
                            disabled={isReadonly}
                            />
                        <span>вопросов в категории</span>
                    </div>
                    {
                        !isReadonly &&
                        <Button
                            cls={b('Add')}
                            theme="normal"
                            size="s"
                            text="Добавить вопрос"
                            onClick={this.onAddQuestion}
                            />
                    }
                </div>
                { this.renderNewQuestions(questions) }
                { this.renderOldQuestions(questions) }
            </div>
        );
    }
}

export default ExamQuestions;
