import map from 'lodash/map';

import { cn } from '@bem-react/classname';
import React, { Component } from 'react';
import {
    Button,
    Icon,
    TextInput
} from 'lego-on-react';
import ExamCategory from 'client/components/examCategory';
import ExamElementRestore from 'client/components/examElementRestore';

import 'lego-on-react/src/components/icon/icon.css';
import 'lego-on-react/src/components/textinput/textinput.css';

import './examSection.css';

import { TEditSection } from 'common/types/editTestData';
import {
    IAddCategoryPayload,
    IEditSectionAllowedFailsPayload,
    IToggleRemoveSectionPayload
} from 'client/store/examData/types';
import { categoriesHasChanges } from 'client/store/examData/helpers';

const b = cn('ExamSection');

interface IExamSectionProps {
    section: TEditSection,
    isReadonly: boolean,
    editSectionAllowedFails(data: IEditSectionAllowedFailsPayload): void,
    toggleRemoveSection(data: IToggleRemoveSectionPayload): void,
    addCategory(data: IAddCategoryPayload): void
}

interface IExamSectionState {
    isExpanded: boolean
}

class ExamSection extends Component<IExamSectionProps, IExamSectionState> {
    constructor(props: IExamSectionProps) {
        super(props);

        this.state = { isExpanded: false };
    }

    private toggleExpand = () => {
        const { isExpanded } = this.state;

        this.setState({ isExpanded: !isExpanded });
    };

    private onAllowedFailsChange = (value: string) => {
        const { section: { code }, editSectionAllowedFails } = this.props;
        const allowedFails = Number(value);

        if (allowedFails >= 0) {
            editSectionAllowedFails({
                sectionCode: code,
                allowedFails
            });
        }
    };

    private remove = () => {
        const { section: { code }, toggleRemoveSection } = this.props;

        toggleRemoveSection({
            sectionCode: code,
            isRemoved: true
        });
    };

    private restore = () => {
        const { section: { code }, toggleRemoveSection } = this.props;

        toggleRemoveSection({
            sectionCode: code,
            isRemoved: false
        });
    };

    private onAddCategory = () => {
        const { section: { code }, addCategory } = this.props;

        addCategory({ sectionCode: code });
    };

    private readonly hasChanges = () => {
        const { section: { isNew, isChanged, categories } } = this.props;

        return isNew || isChanged || categoriesHasChanges(categories);
    };

    public render() {
        const { section, isReadonly } = this.props;
        const { isExpanded } = this.state;
        const { title, code, allowedFails, isRemoved } = section;
        const hasChanges = this.hasChanges();

        if (isRemoved) {
            return (
                <div className={b({ removed: true })}>
                    <ExamElementRestore
                        text="Восстановить секцию"
                        restoreAction={this.restore}
                        />
                </div>
            );
        }

        return (
            <div className={b({ expanded: isExpanded, unsaved: hasChanges })}>
                <div
                    className={b('Remove', { hidden: isReadonly })}
                    onClick={this.remove}
                    >
                    <Icon cls={b('RemoveIcon')} type="gray-cross" />
                </div>

                <div className={b('Wrapper')}>
                    <div
                        className={b('Title')}
                        onClick={this.toggleExpand}
                        >
                        <span>{title}</span>
                        <span className={b('Expand')}>
                            { hasChanges &&
                                <span className={b('Unsaved')} title="Неопубликованные изменения">
                                    <Icon type="dot" />
                                </span>
                            }
                            <Icon type="arrow" />
                        </span>
                    </div>
                    <div className={b('Content')}>
                        <div className={b('Info')}>
                            <label className={b('Label')}>
                                <span className={b('LabelText')}>Код секции</span>
                                <TextInput
                                    cls={b('Input')}
                                    size="s"
                                    theme="normal"
                                    disabled
                                    pin="round-round"
                                    text={code}
                                    />
                            </label>
                            <label className={b('Label')}>
                                <span className={b('LabelText')}>Ошибок в секции</span>
                                <TextInput
                                    cls={`${b('Input')} ${b('Fails')}`}
                                    size="s"
                                    theme="normal"
                                    disabled={isReadonly}
                                    type="number"
                                    pin="round-round"
                                    text={allowedFails}
                                    onChange={this.onAllowedFailsChange}
                                    />
                            </label>
                        </div>
                        <div className={b('Categories')}>
                            {
                                map(section.categories, category => {
                                    return (
                                        <ExamCategory
                                            key={category.id}
                                            category={category}
                                            sectionCode={section.code}
                                            isReadonly={isReadonly}
                                            />
                                    );
                                })
                            }
                            {
                                !isReadonly &&
                                <Button
                                    cls={b('Add')}
                                    theme="normal"
                                    size="s"
                                    text="Добавить категорию"
                                    onClick={this.onAddCategory}
                                    />
                            }
                        </div>
                    </div>
                </div>
            </div>
        );
    }
}

export default ExamSection;
