import map from 'lodash/map';

import { cn } from '@bem-react/classname';
import React, { Component } from 'react';
import { Button, Icon, TextInput } from 'lego-on-react';

import './examSections.css';
import 'lego-on-react/src/components/button/button.css';

import IEditTestData, { TEditSection } from 'common/types/editTestData';
import { IAddSectionPayload } from 'client/store/examData/types';
import ExamSection from 'client/components/examSection';
import { toastr } from 'react-redux-toastr';

const b = cn('ExamSections');

interface IExamSectionsProps {
    editData: IEditTestData,
    isReadonly: boolean,
    questionsPopupRef: any,
    addSection(data: IAddSectionPayload): void
}

interface IExamSectionsState {
    title: string,
    code: string,
    allowedFails: number
}

class ExamSections extends Component<IExamSectionsProps, IExamSectionsState> {
    constructor(props: IExamSectionsProps) {
        super(props);

        this.state = { title: '', code: '', allowedFails: 0 };
    }

    private readonly onAddSection = () => {
        const { editData, addSection } = this.props;
        const { title, code, allowedFails } = this.state;

        const sectionCodes = Object.keys(editData);
        const { serviceId } = editData[sectionCodes[0]];

        const isAlreadyUsed = sectionCodes.some(sectionCode => {
            return sectionCode === code;
        });

        if (isAlreadyUsed) {
            toastr.error('Такой код секции уже существует', '');

            return;
        }

        addSection({
            sectionCode: code,
            title,
            allowedFails,
            serviceId
        });

        this.setState({
            title: '',
            code: '',
            allowedFails: 0
        });
    };

    private readonly onAllowedFailsChange = (value: string) => {
        const allowedFails = Number(value);

        if (allowedFails >= 0) {
            this.setState({ allowedFails });
        }
    };

    private readonly onTitleChange = (value: string) => {
        this.setState({ title: value });
    };

    private readonly onCodeChange = (value: string) => {
        this.setState({ code: value });
    };

    public render() {
        const { editData, isReadonly, questionsPopupRef } = this.props;
        const { title, code, allowedFails } = this.state;
        const isAddDisabled = !title || !code;

        return (
            <div className={b()} ref={questionsPopupRef}>
                {
                    !isReadonly &&
                    <div className={b('AddForm')}>
                        <div className={b('Add')}>
                            <Button
                                size="m"
                                theme="normal"
                                icon={<Icon type="plus-gray" />}
                                onClick={this.onAddSection}
                                disabled={isAddDisabled}
                                />
                        </div>
                        <div className={b('Inputs')}>
                            <TextInput
                                cls={`${b('Input')} ${b('TitleInput')} `}
                                size="s"
                                theme="normal"
                                pin="round-round"
                                text={title}
                                placeholder="Введите название секции"
                                onChange={this.onTitleChange}
                                />
                            <TextInput
                                cls={`${b('Input')} ${b('CodeInput')}`}
                                size="s"
                                theme="normal"
                                pin="round-round"
                                text={code}
                                placeholder="Код секции"
                                onChange={this.onCodeChange}
                                />

                            <label className={b('Label')}>
                                <span className={b('LabelText')}>Ошибок в секции</span>
                                <TextInput
                                    cls={`${b('Input')} ${b('Fails')}`}
                                    size="s"
                                    theme="normal"
                                    type="number"
                                    pin="round-round"
                                    text={allowedFails}
                                    onChange={this.onAllowedFailsChange}
                                    />
                            </label>
                        </div>
                    </div>
                }
                <div className={b('Content')}>
                    {
                        map(editData, (section: TEditSection) => {
                            return (
                                <ExamSection
                                    key={section.code}
                                    section={section}
                                    isReadonly={isReadonly}
                                    />
                            );
                        })
                    }
                </div>
            </div>
        );
    }
}

export default ExamSections;
