import { cn, classnames as classNames } from '@bem-react/classname';
import React, { Component } from 'react';

import { Button, Select } from 'lego-on-react';

import 'lego-on-react/src/components/button/button.css';
import 'lego-on-react/src/components/select/select.css';
import '../reportBlock/reportBlock.css';
import './freezing.css';
import Dictionary from 'common/types/dictionary';

interface IFreezingProps {
    frozenExams: number[],
    availableForFreezeExams: number[],
    examsById: Dictionary<string>,
    lastAttemptFinish?: string | null,
    isUnfrozenSuccess?: boolean,
    isError?: boolean,
    examsForFreeze: number[],
    examsForUnfreeze: number[],
    chooseForFreezeAction(exams: number[]): void,
    chooseForUnfreezeAction(exams: number[]): void,
    freezeAction(exams: number[]): void,
    unfreezeAction(exams: number[]): void
}

const b = cn('Freezing');
const reportBlock = cn('ReportBlock');

class Freezing extends Component<IFreezingProps> {
    private _getMessage(lastAttemptFinishDate?: Date, isUnfrozenSuccess?: boolean) {
        if (lastAttemptFinishDate) {
            return (
                <div className={b('Message')}>
                    Тесты заморожены.
                    <br />
                    Последняя попытка завершится: {lastAttemptFinishDate.toString()}
                </div>
            );
        }

        if (isUnfrozenSuccess) {
            return <div className={b('Message')}>Тесты открыты для прохождения</div>;
        }
    }

    private _chooseForFreeze = (exams: any[]) => {
        this.props.chooseForFreezeAction(exams);
    }

    private _chooseForUnfreeze = (exams: any[]) => {
        this.props.chooseForUnfreezeAction(exams);
    }

    private _onFreezeClick = () => {
        this.props.freezeAction(this.props.examsForFreeze);
    };

    private _onUnfreezeClick = () => {
        this.props.unfreezeAction(this.props.examsForUnfreeze);
    };

    private _selectAllForFreeze = () => {
        this.props.chooseForFreezeAction(this.props.availableForFreezeExams);
    }

    private _selectAllForUnfreeze = () => {
        this.props.chooseForUnfreezeAction(this.props.frozenExams);
    }

    public render() {
        const {
            isUnfrozenSuccess,
            lastAttemptFinish,
            isError,
            frozenExams,
            availableForFreezeExams,
            examsById,
            examsForFreeze = [],
            examsForUnfreeze = []
        } = this.props;

        let lastAttemptFinishDate;

        if (lastAttemptFinish) {
            lastAttemptFinishDate = new Date(lastAttemptFinish);
        }

        return (
            <div className={classNames(b(), reportBlock())}>
                <div className={classNames(b('Title'), reportBlock('Title'))}>
                    Закрытие сервиса для прохождения тестирования
                </div>
                <div className={classNames(b('Controls'), reportBlock('Controls'))}>
                    <div className={classNames(b('Control'))}>
                        <Select
                            theme="normal"
                            size="m"
                            type="check"
                            text="vary"
                            width="max"
                            name="selectFreeze"
                            placeholder="Закрыть"
                            val={examsForFreeze}
                            disabled={!availableForFreezeExams.length}
                            onChange={this._chooseForFreeze}
                            >
                            {
                                availableForFreezeExams.map((id: number) => {
                                    const slug = examsById[id];

                                    return (
                                        <Select.Item key={id} val={id}>
                                            {slug}
                                        </Select.Item>
                                    );
                                })}
                        </Select>
                        <Button
                            cls={b('Choose')}
                            theme="normal"
                            size="m"
                            text="Выбрать все"
                            disabled={!availableForFreezeExams.length}
                            onClick={this._selectAllForFreeze}
                            />
                        <div className={b('Action')}>
                            <Button
                                theme="action"
                                size="m"
                                text="Закрыть доступ"
                                onClick={this._onFreezeClick}
                                disabled={!examsForFreeze.length}
                                />
                        </div>
                    </div>
                    <div className={classNames(b('Control'))}>
                        <Select
                            theme="normal"
                            size="m"
                            type="check"
                            text="vary"
                            width="max"
                            name="selectUnfreeze"
                            placeholder="Открыть"
                            val={examsForUnfreeze}
                            disabled={!frozenExams.length}
                            onChange={this._chooseForUnfreeze}
                            >
                            {
                                frozenExams.map((id: number) => {
                                    const slug = examsById[id];

                                    return (
                                        <Select.Item key={id} val={id}>
                                            {slug}
                                        </Select.Item>
                                    );
                                })}
                        </Select>
                        <Button
                            cls={b('Choose')}
                            theme="normal"
                            size="m"
                            text="Выбрать все"
                            disabled={!frozenExams.length}
                            onClick={this._selectAllForUnfreeze}
                            />
                        <div className={b('Action')}>
                            <Button
                                theme="action"
                                size="m"
                                text="Открыть доступ"
                                onClick={this._onUnfreezeClick}
                                disabled={!examsForUnfreeze.length}
                                />
                        </div>
                    </div>
                </div>
                { this._getMessage(lastAttemptFinishDate, isUnfrozenSuccess) }
                { isError &&
                    <div className={b('Error')}>Что-то пошло не так, попробуйте снова</div>
                }
            </div>
        );
    }
}

export default Freezing;
