import { cn } from '@bem-react/classname';
import React, { Component, Fragment } from 'react';
import { Button, Spin } from 'lego-on-react';
import { toastr } from 'react-redux-toastr';

import StatisticsInput from 'client/components/statisticsInput';
import ConfirmationModal from 'client/components/confirmationModal/confirmationModal';
import AttemptsInfo from 'client/components/attemptsInfo';
import { IAttemptInfo } from 'client/store/nullifyAttempt/types';

import { getDetailedError, IDetailsError } from 'common/utils/error';

import './nullifyAttempt.css';
import 'lego-on-react/src/components/button/button.css';
import 'lego-on-react/src/components/spin/spin.css';

const b = cn('NullifyAttempt');

interface INullifyAttemptProps {
    attemptIds: string,
    hasFocusAttemptIds: boolean,
    isValidAttemptIds: boolean,
    hasAttemptIds: boolean,
    isPending: boolean,
    requestError: IDetailsError | null,
    isVisibleConfirmationModal: boolean,
    attemptsInfo: IAttemptInfo[],
    isNullifySuccess: boolean,
    notFoundAttempts: number[] | null,
    typeAttemptIdsAction(inputText: string): void,
    focusAttemptIds(): void,
    blurAttemptIds(): void,
    getAttemptsInfo(attemptIds: string): void,
    showConfirmation(): void,
    decline(): void,
    confirm(attemptIds: string): void
}

class NullifyAttempt extends Component<INullifyAttemptProps> {
    private getAttemptIdsErrorMessage() {
        const { isValidAttemptIds } = this.props;

        if (!isValidAttemptIds) {
            return 'Введите числа через запятую';
        }

        return '';
    }

    private readonly showError = (error: IDetailsError) => {
        const { message, details } = getDetailedError(error);

        toastr.error(message, details);
    };

    private readonly getAttemptsInfo = () => {
        const { attemptIds, getAttemptsInfo } = this.props;

        if (attemptIds) {
            return getAttemptsInfo(attemptIds);
        }
    };

    private readonly confirm = () => {
        const { attemptIds, confirm } = this.props;

        confirm(attemptIds);
    };

    private readonly getNullifySuccessMessage = () => {
        const { isNullifySuccess, notFoundAttempts } = this.props;

        if (!isNullifySuccess) {
            return;
        }

        if (notFoundAttempts && notFoundAttempts.length) {
            const message = `Не найдены попытки ${notFoundAttempts.join(', ')}`;

            return <div className={b('Success')}>{message}</div>;
        }
    };

    public componentDidUpdate() {
        const {
            requestError,
            isNullifySuccess
        } = this.props;

        if (requestError) {
            this.showError(requestError);
        }

        if (isNullifySuccess) {
            toastr.success('Попытки аннулированы', '');
        }
    }

    private readonly renderAttempts = (disabled: boolean) => {
        const { attemptsInfo, showConfirmation } = this.props;
        const hasInfo = attemptsInfo && attemptsInfo.length;

        if (!attemptsInfo) {
            return;
        }

        return (
            <Fragment>
                {
                    hasInfo ? <AttemptsInfo attemptsInfo={attemptsInfo} /> :
                    <div className={b('NotFound')}>Попытки не найдены</div>
                }
                { hasInfo ? <Button
                    theme="action"
                    size="m"
                    type="submit"
                    text="Аннулировать"
                    onClick={showConfirmation}
                    disabled={disabled}
                    /> : null
                }
            </Fragment>
        );
    };

    public render() {
        const {
            attemptIds,
            hasFocusAttemptIds,
            isValidAttemptIds,
            typeAttemptIdsAction,
            focusAttemptIds,
            blurAttemptIds,
            isPending,
            isVisibleConfirmationModal,
            decline
        } = this.props;

        const disabled = isPending || !isValidAttemptIds;

        return (
            <div className={b()}>
                <div className={b('Title')}>Аннулирование попыток</div>
                <div className={b('Input')}>
                    <StatisticsInput
                        tip="Номера попыток"
                        buttonText="Получить информацию"
                        inputText={attemptIds}
                        hasFocus={hasFocusAttemptIds}
                        placeholder="1, 2"
                        errorMessage={this.getAttemptIdsErrorMessage()}
                        typeAction={typeAttemptIdsAction}
                        focus={focusAttemptIds}
                        blur={blurAttemptIds}
                        onSubmit={this.getAttemptsInfo}
                        disabled={disabled}
                        />
                </div>
                { this.renderAttempts(disabled) }
                <ConfirmationModal
                    confirmationText="Вы уверены?"
                    confirmText="Да"
                    declineText="Нет"
                    onConfirm={this.confirm}
                    onDecline={decline}
                    visible={isVisibleConfirmationModal}
                    />
                <div className={b('Loading')}>
                    <Spin size="m" progress={isPending} />
                </div>
                { this.getNullifySuccessMessage() }
            </div>
        );
    }
}

export default NullifyAttempt;
