import { cn } from '@bem-react/classname';
import React, { Component, Fragment } from 'react';
import { Button, Spin } from 'lego-on-react';
import { toastr } from 'react-redux-toastr';

import StatisticsInput from 'client/components/statisticsInput';
import ConfirmationModal from 'client/components/confirmationModal/confirmationModal';
import CertificatesInfo from 'client/components/certificatesInfo';

import './nullifyCertificate.css';
import 'lego-on-react/src/components/button/button.css';
import 'lego-on-react/src/components/spin/spin.css';

import RequestStatus from 'common/types/requestStatus';
import { getDetailedError, IDetailsError } from 'common/utils/error';
import { ICertificatesInfo } from 'client/store/nullifyCertificate/types';

const b = cn('NullifyCertificate');

interface INullifyCertificateProps {
    certIds: string,
    isValidCertIds: boolean,
    hasFocusCertIds: boolean,
    nullifyCertsStatus: RequestStatus | null,
    getInfoStatus: RequestStatus | null,
    requestError: IDetailsError | null,
    isVisibleConfirmationModal: boolean,
    certificatesInfo: ICertificatesInfo[] | null
    typeCertificateIdsAction(inputText: string): void,
    focusCertificateIds(): void,
    blurCertificateIds(): void,
    getCertificatesInfo(certIds: string): void,
    showConfirmation(): void,
    decline(): void,
    confirm(certIds: string): void
}

class NullifyCertificate extends Component<INullifyCertificateProps> {
    private getCertIdsErrorMessage() {
        const { isValidCertIds } = this.props;

        if (!isValidCertIds) {
            return 'Введите числа через запятую';
        }

        return '';
    }

    private readonly showError = (error: IDetailsError) => {
        const { message, details } = getDetailedError(error);

        toastr.error(message, details);
    };

    private readonly getCertsInfo = () => {
        const { certIds, getCertificatesInfo } = this.props;

        if (certIds) {
            return getCertificatesInfo(certIds);
        }
    };

    private readonly confirm = () => {
        const { certIds, confirm } = this.props;

        confirm(certIds);
    };

    private readonly renderCertificates = (disabled: boolean) => {
        const {
            certificatesInfo,
            showConfirmation
        } = this.props;

        const hasInfo = certificatesInfo && certificatesInfo.length;

        if (!certificatesInfo) {
            return;
        }

        return (
            <Fragment>
                {
                    hasInfo ? <CertificatesInfo certificatesInfo={certificatesInfo} /> :
                    <div className={b('NotFound')}>Сертификаты не найдены</div>
                }
                { hasInfo ? <Button
                    cls={b('NullifyButton')}
                    theme="action"
                    size="m"
                    type="submit"
                    text="Аннулировать"
                    onClick={showConfirmation}
                    disabled={disabled}
                    /> : null
                }
            </Fragment>
        );
    };

    public componentDidUpdate() {
        const {
            requestError,
            nullifyCertsStatus
        } = this.props;

        if (requestError) {
            this.showError(requestError);
        }

        if (nullifyCertsStatus === RequestStatus.success) {
            toastr.success('Сертификаты аннулированы', '');
        }
    }

    public render() {
        const {
            certIds,
            isValidCertIds,
            nullifyCertsStatus,
            getInfoStatus,
            isVisibleConfirmationModal,
            typeCertificateIdsAction,
            focusCertificateIds,
            blurCertificateIds,
            decline,
            hasFocusCertIds
        } = this.props;

        const isPending = nullifyCertsStatus === RequestStatus.pending ||
            getInfoStatus === RequestStatus.pending;
        const disabled = isPending || !isValidCertIds;

        return (
            <div className={b()}>
                <div className={b('Title')}>Аннулирование сертификатов</div>
                <div className={b('Input')}>
                    <StatisticsInput
                        tip="Номера сертификатов"
                        buttonText="Получить информацию"
                        inputText={certIds}
                        placeholder="1, 2"
                        errorMessage={this.getCertIdsErrorMessage()}
                        typeAction={typeCertificateIdsAction}
                        hasFocus={hasFocusCertIds}
                        focus={focusCertificateIds}
                        blur={blurCertificateIds}
                        onSubmit={this.getCertsInfo}
                        disabled={disabled}
                        />
                </div>
                { this.renderCertificates(disabled) }
                <ConfirmationModal
                    confirmationText="Вы уверены?"
                    confirmText="Да"
                    declineText="Нет"
                    onConfirm={this.confirm}
                    onDecline={decline}
                    visible={isVisibleConfirmationModal}
                    />
                <div className={b('Loading')}>
                    <Spin size="m" progress={isPending} />
                </div>
            </div>
        );
    }
}

export default NullifyCertificate;
