import { cn } from '@bem-react/classname';
import React, { Component } from 'react';

import DataByCertNumber from 'client/components/dataByCertNumber';
import CertificatesByPeriod from 'client/components/certificatesByPeriod';
import DataByPeriod from 'client/components/dataByPeriod';
import AttemptsByLogin from 'client/components/attemptsByLogin';
import AttemptByNumber from 'client/components/attemptByNumber';
import TrialsByPeriod from 'client/components/trialsByPeriod';
import BansByLogin from 'client/components/bansByLogin';
import BansAggregation from 'client/components/bansAggregation';
import AgenciesStat from 'client/components/agenciesStat';
import QuestionsStatistics from 'client/components/questionsStatistics';
import Dictionary from 'common/types/dictionary';
import { Roles } from 'common/types/adminData';
import { IReportComponentFabric, ReportData } from 'common/types/reports';

interface IReportsProps {
    reportsUrls: Dictionary<string>,
    reportTypes: string[],
    examSlugs: Dictionary<string>,
    intervals: Dictionary<string>,
    agenciesStatUrl: string,
    userRoles: Roles[],
    lastSyncAgenciesDate: string
}

const dataByCertNumber: IReportComponentFabric = (reportData: ReportData, type: string) => {
    return (<DataByCertNumber
        urlTemplate={reportData.reportsUrls.certificate}
        key={type}
        />);
};

const certificatesByPeriod: IReportComponentFabric = (reportData: ReportData, type: string) => {
    return (<CertificatesByPeriod
        urlTemplate={reportData.reportsUrls.certificates}
        examSlugs={reportData.examSlugs}
        key={type}
        />);
};

const dataByPeriod: IReportComponentFabric = (reportData: ReportData, type: string) => {
    return (<DataByPeriod
        urlTemplate={reportData.reportsUrls.certificatesAggregation}
        examSlugs={reportData.examSlugs}
        intervals={reportData.intervals}
        key={type}
        />);
};

const attemptsByLogin: IReportComponentFabric = (reportData: ReportData, type: string) => {
    return (<AttemptsByLogin
        urlTemplate={reportData.reportsUrls.userTrials}
        key={type}
        />);
};

const attemptByNumber: IReportComponentFabric = (reportData: ReportData, type: string) => {
    return (<AttemptByNumber
        urlTemplate={reportData.reportsUrls.trial}
        key={type}
        />);
};

const trialsByPeriod: IReportComponentFabric = (reportData: ReportData, type: string) => {
    return (<TrialsByPeriod
        urlTemplate={reportData.reportsUrls.trials}
        examSlugs={reportData.examSlugs}
        key={type}
        />);
};

const bansByLogin: IReportComponentFabric = (reportData: ReportData, type: string) => {
    return (<BansByLogin
        urlTemplate={reportData.reportsUrls.bansByLogin}
        key={type}
        />);
};

const bansAggregation: IReportComponentFabric = (reportData: ReportData, type: string) => {
    return (<BansAggregation
        urlTemplate={reportData.reportsUrls.certificatesAggregation}
        examSlugs={reportData.examSlugs}
        intervals={reportData.intervals}
        key={type}
        />);
};

const questionsStatistics: IReportComponentFabric = (reportData: ReportData, type: string) => {
    return (<QuestionsStatistics
        urlTemplate={reportData.reportsUrls.questions}
        examSlugs={reportData.examSlugs}
        key={type}
        />);
};

const reportTypeToComponent: Dictionary<IReportComponentFabric> = {
    certificate: dataByCertNumber,
    certificates: certificatesByPeriod,
    certificatesAggregation: dataByPeriod,
    userTrials: attemptsByLogin,
    trial: attemptByNumber,
    trials: trialsByPeriod,
    bansByLogin,
    bansAggregation,
    questions: questionsStatistics
};

const b = cn('Reports');

class Reports extends Component<IReportsProps> {
    public render() {
        const {
            reportsUrls,
            reportTypes,
            examSlugs,
            intervals,
            agenciesStatUrl,
            userRoles,
            lastSyncAgenciesDate
        } = this.props;

        const reportData: ReportData = {
            examSlugs,
            intervals,
            reportsUrls
        };

        const isAnalyst = userRoles.includes(Roles.analyst);

        return (
            <div className={b()}>
                {
                    reportTypes.map(type => (
                        reportTypeToComponent[type] &&
                        reportTypeToComponent[type](reportData, type)
                    ))
                }
                {
                    isAnalyst && <AgenciesStat
                        urlTemplate={agenciesStatUrl}
                        lastSyncAgenciesDate={lastSyncAgenciesDate}
                        />
                }
            </div>
        );
    }
}

export default Reports;
