import { cn } from '@bem-react/classname';
import React, { createRef, PureComponent } from 'react';

import { Button, TextInput, Tooltip } from 'lego-on-react';

import 'lego-on-react/src/components/button/button.css';
import 'lego-on-react/src/components/tooltip/tooltip.css';
import 'lego-on-react/src/components/textinput/textinput.css';

import './statisticsInput.css';

const b = cn('StatisticsInput');

export interface IStatisticsInputProps {
    tip?: string,
    buttonText: string,
    inputText?: string,
    hasFocus?: boolean,
    inputId?: number,
    placeholder?: string,
    errorMessage?: string,
    disabled?: boolean
    onSubmit(): void,
    typeAction(inputText: string): void,
    focus(): void,
    blur(): void
}

class StatisticsInput extends PureComponent<IStatisticsInputProps> {
    private readonly _inputRef = createRef<TextInput>();

    public componentDidMount() {
        if (window) {
            window.addEventListener('keydown', this._onKeyDown);
        }
    }

    public componentWillUnmount() {
        if (window) {
            window.removeEventListener('keydown', this._onKeyDown);
        }
    }

    public render() {
        const {
            tip,
            buttonText,
            inputText = '',
            inputId = 1,
            placeholder = '',
            errorMessage,
            disabled,
            onSubmit,
            focus,
            blur
        } = this.props;

        const inputIdString = inputId.toString();

        return (
            <div className={b()}>
                { tip && (<label className={b('Tip')} htmlFor={inputIdString}>{tip}</label>) }
                <div className={b('Input')}>
                    <TextInput
                        id={inputIdString}
                        ref={this._inputRef}
                        theme="normal"
                        size="m"
                        text={inputText}
                        placeholder={placeholder}
                        onChange={this._onInputChange}
                        onFocus={focus}
                        onBlur={blur}
                        />
                    <Tooltip
                        anchor={this._inputRef.current}
                        theme="error"
                        size="m"
                        visible={Boolean(errorMessage)}
                        >
                        {errorMessage}
                    </Tooltip>
                </div>
                <div className={b('Button')}>
                    <Button
                        theme="action"
                        size="m"
                        text={buttonText}
                        disabled={disabled || !inputText}
                        onClick={onSubmit}
                        />
                </div>
            </div>
        );
    }

    private readonly _onInputChange = (inputText: string) => {
        const { typeAction } = this.props;

        typeAction(inputText);
    };

    private readonly _onKeyDown = ({ key }: KeyboardEvent) => {
        const { hasFocus = false, onSubmit } = this.props;

        if (hasFocus && key === 'Enter') {
            onSubmit();
        }
    };
}

export default StatisticsInput;
