import { cn } from '@bem-react/classname';

import React, { Component } from 'react';
import { Button, CheckBox, Select, Spin, TextArea, TextInput } from 'lego-on-react';
import ConfirmationModal from 'client/components/confirmationModal';
import Dictionary from 'common/types/dictionary';
import { toastr } from 'react-redux-toastr';

import './unbanForm.css';
import 'lego-on-react/src/components/button/button.css';
import 'lego-on-react/src/components/checkbox/checkbox.css';
import 'lego-on-react/src/components/select/select.css';
import 'lego-on-react/src/components/spin/spin.css';
import 'lego-on-react/src/components/textarea/textarea.css';
import 'lego-on-react/src/components/textinput/textinput.css';

import { IUnbanPayload } from 'client/store/unbanForm/types';
import { IGlobalUserInfo } from 'client/store/bans/types';
import RequestStatus from 'common/types/requestStatus';
import { getDetailedError, IDetailsError } from 'common/utils/error';

const b = cn('UnbanForm');

interface UnbanFormProps {
    examsById: Dictionary<string>,
    globalUserInfo: IGlobalUserInfo,
    examsForUnban: string[],
    userLogin: string,
    banLogin: string,
    unbanStatus: RequestStatus | null,
    requestError: IDetailsError | null,
    submitErrors: string[] | null,
    isSuperUnban: boolean,
    reason: string,
    isVisibleConfirmationModal: boolean,
    bannedExamsIds: string[],
    chooseExams(exams: string[]): void,
    typeLogin(inputText: string): void,
    typeReason(inputText: string): void,
    toggleSuperUnban(): void,
    unban(unbanData: IUnbanPayload): void,
    pressUnban(): void,
    decline(): void,
    getBanInfo(login: string): void
}

class UnbanForm extends Component<UnbanFormProps> {
    private readonly showError = (error: IDetailsError) => {
        const { message, details } = getDetailedError(error);

        toastr.error(message, details);
    };

    private readonly onUnbanClick = () => {
        const { pressUnban } = this.props;

        pressUnban();
    };

    private readonly confirmUnban = () => {
        const {
            userLogin,
            examsForUnban,
            reason,
            isSuperUnban,
            unban,
            examsById
        } = this.props;

        let examIds = examsForUnban;

        if (isSuperUnban) {
            examIds = Object.keys(examsById);
        }

        return unban({
            userLogin,
            examsForUnban: examIds,
            reason,
            isSuperUnban
        });
    };

    private readonly showFormValidationError = () => {
        const { submitErrors } = this.props;
        const error = submitErrors!.join('; ');

        toastr.error('Проверьте поля формы', error);
    };

    public componentDidUpdate() {
        const {
            requestError,
            userLogin,
            unbanStatus,
            getBanInfo,
            submitErrors
        } = this.props;

        if (requestError) {
            this.showError(requestError);
        }

        if (unbanStatus === RequestStatus.success) {
            toastr.success('Пользователь разбанен', '');

            if (userLogin) {
                getBanInfo(userLogin);
            }
        }

        if (submitErrors) {
            this.showFormValidationError();
        }
    }

    public componentDidMount() {
        const { banLogin, typeLogin } = this.props;

        if (banLogin) {
            typeLogin(banLogin);
        }
    }

    public render() {
        const {
            examsForUnban,
            examsById,
            userLogin,
            reason,
            isSuperUnban,
            toggleSuperUnban,
            chooseExams,
            unbanStatus,
            isVisibleConfirmationModal,
            decline,
            typeReason,
            globalUserInfo,
            typeLogin,
            bannedExamsIds
        } = this.props;

        const isPending = unbanStatus === RequestStatus.pending;
        const isUserSuperbanned = globalUserInfo.isBanned;
        const isChoiceDisabled = isPending
            || bannedExamsIds.length === 0
            || isSuperUnban
            || isUserSuperbanned;

        return (
            <div className={b()}>
                <form>
                    <div className={b('Title')}>Разбанить</div>
                    <label htmlFor="login" className={b('Label')}>
                        Логин для разбана
                        <span className={b('RequiredMark')}>*</span>
                    </label>
                    <TextInput
                        cls={b('Field')}
                        theme="normal"
                        size="m"
                        name="login"
                        placeholder="Логин для разбана"
                        text={userLogin}
                        onChange={typeLogin}
                        disabled={isPending}
                        />
                    <Select
                        theme="normal"
                        cls={b('Field')}
                        size="m"
                        type="check"
                        text="vary"
                        width="max"
                        name="selectUnban"
                        placeholder="Тесты для разбана"
                        val={examsForUnban}
                        disabled={isChoiceDisabled}
                        onChange={chooseExams}
                        >
                        {
                            bannedExamsIds.map((id: string) => {
                                const slug = examsById[id];

                                return (
                                    <Select.Item key={id} val={id}>
                                        {slug}
                                    </Select.Item>
                                );
                            })}
                    </Select>
                    <label htmlFor="reason" className={b('Label')}>
                        Причина разбана<span className={b('RequiredMark')}>*</span>
                    </label>
                    <TextArea
                        cls={b('Field')}
                        theme="normal"
                        placeholder="Причина разбана"
                        size="m"
                        name="reason"
                        rows={2}
                        text={reason}
                        onChange={typeReason}
                        disabled={isPending}
                        />
                    <CheckBox
                        cls={b('Field')}
                        theme="normal"
                        size="m"
                        text="Суперразбан"
                        checked={isSuperUnban}
                        onChange={toggleSuperUnban}
                        disabled={!isUserSuperbanned || isPending}
                        />
                    <div className={b('UnbanActions')}>
                        <Button
                            cls={b('UnbanButton')}
                            theme="action"
                            size="m"
                            text="Разбанить"
                            onClick={this.onUnbanClick}
                            disabled={isPending}
                            />
                        <div className={b('Loading')}>
                            <Spin size="m" progress={isPending} />
                        </div>
                    </div>
                </form>
                <ConfirmationModal
                    confirmationText="Вы уверены?"
                    confirmText="Да"
                    declineText="Нет"
                    onConfirm={this.confirmUnban}
                    onDecline={decline}
                    visible={isVisibleConfirmationModal}
                    />
            </div>
        );
    }
}

export default UnbanForm;
