import { cn } from '@bem-react/classname';
import React, { Component } from 'react';
import { toastr } from 'react-redux-toastr';

import StatisticsInput from '../statisticsInput';
import VideosContent from '../videosContent/videosContent';

import {
    VerdictType,
    IVideosError,
    IAppealDataFromDB,
    IProctoringResponsesInfo
} from '../../store/videos/types';

import './videos.css';

import ERROR_CODES from '../../../common/types/errorCodes';
import RequestStatus from '../../../common/types/requestStatus';

interface IVideosProps {
    attemptId: string,
    hasFocusAttemptId: boolean,
    isValidAttemptId: boolean,
    hasAttemptId: boolean,
    videos: string[],
    verdict: VerdictType,
    email: string,
    hasFocusEmail: boolean,
    isValidEmail: boolean,
    isSubmitError: boolean,
    hasEmail: boolean,
    requestStatus: RequestStatus | null,
    requestError: IVideosError | null,
    isVisibleConfirmationModal: boolean,
    isAppealSuccess: boolean,
    appealDataFromDB: IAppealDataFromDB | null,
    pdfLink: string,
    proctoringResponsesInfo: IProctoringResponsesInfo[] | null,
    typeAttemptIdAction(inputText: string): void,
    focusAttemptId(): void,
    blurAttemptId(): void,
    getVideos(attemptId: string): void,
    decide(verdict: VerdictType): void,
    typeEmailAction(inputText: string): void,
    focusEmail(): void,
    blurEmail(): void,
    showConfirmation(): void,
    confirm(attemptId: string, verdict: VerdictType, email: string): void,
    decline(): void
}

const requiredFieldErrorMessage = 'Это поле обязательно для заполнения';

const b = cn('Videos');

class Videos extends Component<IVideosProps> {
    private _showError = (error: IVideosError) => {
        const { internalCode } = error;
        const errorHandler = ERROR_CODES[internalCode] || ERROR_CODES.default;

        toastr.error(errorHandler.text, '');
    };

    public componentDidUpdate() {
        const { requestError, isAppealSuccess } = this.props;

        if (requestError) {
            this._showError(requestError);
        }

        if (isAppealSuccess) {
            toastr.success('Вердикт сохранён', '');
        }
    }

    public render() {
        const {
            videos,
            verdict,
            attemptId,
            hasFocusAttemptId,
            isSubmitError,
            typeAttemptIdAction,
            focusAttemptId,
            blurAttemptId,
            requestStatus,
            requestError,
            email,
            hasFocusEmail,
            isValidEmail,
            hasEmail,
            typeEmailAction,
            focusEmail,
            blurEmail,
            showConfirmation,
            decide,
            isVisibleConfirmationModal,
            confirm,
            decline,
            isAppealSuccess,
            isValidAttemptId,
            appealDataFromDB,
            pdfLink,
            proctoringResponsesInfo
        } = this.props;

        const attemptIdErrorMessage = isSubmitError ?
            requiredFieldErrorMessage :
            this._getAttemptIdErrorMessage();

        const isVideosContentVisible = !isAppealSuccess && requestStatus !== null;

        return (
            <div className={b()}>
                <div className={b('Input')}>
                    <StatisticsInput
                        tip="Номер попытки"
                        buttonText="Получить видео"
                        inputText={attemptId}
                        hasFocus={hasFocusAttemptId}
                        placeholder="12345"
                        errorMessage={attemptIdErrorMessage}
                        typeAction={typeAttemptIdAction}
                        focus={focusAttemptId}
                        blur={blurAttemptId}
                        onSubmit={this._getVideos}
                        />
                </div>
                {
                    isVideosContentVisible && (
                        <VideosContent
                            attemptId={attemptId}
                            isValidAttemptId={isValidAttemptId}
                            videos={videos}
                            verdict={verdict || ''}
                            email={email}
                            hasFocusEmail={hasFocusEmail}
                            isValidEmail={isValidEmail}
                            hasEmail={hasEmail}
                            requestError={requestError}
                            isSubmitError={isSubmitError}
                            requestStatus={requestStatus}
                            isVisibleConfirmationModal={isVisibleConfirmationModal}
                            requiredFieldErrorMessage={requiredFieldErrorMessage}
                            appealDataFromDB={appealDataFromDB}
                            pdfLink={pdfLink}
                            decide={decide}
                            typeEmailAction={typeEmailAction}
                            focusEmail={focusEmail}
                            blurEmail={blurEmail}
                            showConfirmation={showConfirmation}
                            confirm={confirm}
                            decline={decline}
                            proctoringResponsesInfo={proctoringResponsesInfo}
                            />
                    )
                }
            </div>
        );
    }

    private _getAttemptIdErrorMessage() {
        const { isValidAttemptId, hasAttemptId } = this.props;

        if (!isValidAttemptId) {
            return 'Введите число';
        }

        if (!hasAttemptId) {
            return requiredFieldErrorMessage;
        }

        return '';
    }

    private readonly _getVideos = () => {
        const { attemptId, getVideos } = this.props;

        if (attemptId) {
            getVideos(attemptId);
        }
    };
}

export default Videos;
