import { cn } from '@bem-react/classname';
import React, { ChangeEvent, Component, Fragment } from 'react';

import { Button, RadioButton, Spin } from 'lego-on-react';

import { validateAppealData } from 'client/common/validate';

import TolokaInfo from 'client/components/tolokaInfo';
import Video from 'client/components/video';
import StatisticsInput from 'client/components/statisticsInput';
import ConfirmationModal from 'client/components/confirmationModal/confirmationModal';
import Dictionary from 'common/types/dictionary';
import RequestStatus from 'common/types/requestStatus';

import 'lego-on-react/src/components/spin/spin.css';

import {
    IAppealDataFromDB,
    IProctoringResponsesInfo,
    IVideosError,
    VerdictType
} from '../../store/videos/types';

import './videosContent.css';

interface IVideosContentProps {
    attemptId: string,
    isValidAttemptId: boolean,
    videos: string[],
    verdict: VerdictType,
    email: string,
    hasFocusEmail: boolean,
    isValidEmail: boolean,
    hasEmail: boolean,
    isSubmitError: boolean,
    requestStatus: RequestStatus | null,
    requestError: IVideosError | null,
    isVisibleConfirmationModal: boolean,
    requiredFieldErrorMessage: string,
    appealDataFromDB: IAppealDataFromDB | null,
    pdfLink: string,
    proctoringResponsesInfo: IProctoringResponsesInfo[] | null,
    decide(verdict: VerdictType): void,
    typeEmailAction(inputText: string): void,
    focusEmail(): void,
    blurEmail(): void,
    showConfirmation(): void,
    confirm(attemptId: string, verdict: VerdictType, email: string): void,
    decline(): void
}

const alreadyAppealed = 'По данной попытке уже была проведена апелляция. Вердикт: ';

const appealTexts: Dictionary<string> = {
    correct: 'нет нарушений',
    failed: 'есть нарушения'
};

const b = cn('VideosContent');

class VideosContent extends Component<IVideosContentProps> {
    /* eslint-disable complexity */
    public render() {
        const {
            videos,
            requestStatus,
            email,
            hasFocusEmail,
            typeEmailAction,
            focusEmail,
            blurEmail,
            showConfirmation,
            verdict,
            isVisibleConfirmationModal,
            decline,
            isSubmitError,
            requiredFieldErrorMessage,
            attemptId,
            isValidAttemptId,
            isValidEmail,
            pdfLink,
            proctoringResponsesInfo
        } = this.props;

        const hasVideos = videos.length > 0;
        const hasAdditionalInfo = proctoringResponsesInfo && proctoringResponsesInfo.length > 0;
        const emailErrorMessage = isSubmitError ?
            requiredFieldErrorMessage :
            this._getEmailErrorMessage();

        const isAppealDataCorrect = validateAppealData({
            attemptId,
            isValidAttemptId,
            email,
            isValidEmail,
            verdict
        });

        const appealMessage = this._getAppealMessage();
        const isAppealButtonDisabled = !isAppealDataCorrect ||
            requestStatus === RequestStatus.pending;

        return (
            <div className={b()}>
                {
                    pdfLink && (
                        <div className={b('PdfButton')}>
                            <Button
                                theme="normal"
                                title="PDF-отчет по попытке"
                                size="m"
                                type="link"
                                text="Скачать pdf-отчет"
                                url={pdfLink}
                                target="blank"
                                />
                        </div>
                    )
                }
                {
                    hasVideos ? (
                        <Fragment>
                            <div className={b('List')}>
                                {
                                    videos.map(src => (
                                        <Video
                                            key={src}
                                            src={src}
                                            />
                                    ))
                                }
                            </div>
                            {
                                hasAdditionalInfo &&
                                <TolokaInfo tolokaInfo={proctoringResponsesInfo!} />
                            }
                        </Fragment>
                    ) : <div className={b('NoVideo')}>Нет сохраненных видео для данной попытки</div>
                }
                {
                    appealMessage || (
                        <div className={b('Verdict')}>
                            <div className={b('Decision')}>
                                <RadioButton
                                    size="m"
                                    theme="normal"
                                    name="decision"
                                    value={verdict || undefined}
                                    onChange={this._onDecision}
                                    >
                                    <RadioButton.Radio value="correct">
                                        Создать сертификат
                                    </RadioButton.Radio>
                                    <RadioButton.Radio value="failed">
                                        Есть нарушения
                                    </RadioButton.Radio>
                                </RadioButton>
                            </div>
                            <div className={b('Input')}>
                                <StatisticsInput
                                    tip="Email"
                                    buttonText="Отправить"
                                    inputText={email}
                                    hasFocus={hasFocusEmail}
                                    placeholder="expert-dev@yandex-team.ru"
                                    errorMessage={emailErrorMessage}
                                    typeAction={typeEmailAction}
                                    focus={focusEmail}
                                    blur={blurEmail}
                                    onSubmit={showConfirmation}
                                    disabled={isAppealButtonDisabled}
                                    />
                            </div>
                        </div>
                    )
                }
                <ConfirmationModal
                    confirmationText="Вы уверены?"
                    confirmText="Да"
                    declineText="Нет"
                    onConfirm={this._confirm}
                    onDecline={decline}
                    visible={isVisibleConfirmationModal}
                    />
                <div className={b('Loading')}>
                    <Spin size="m" progress={requestStatus === RequestStatus.pending} />
                </div>
            </div>
        );
    }
    /* eslint-disable complexity */

    private _onDecision = (event: ChangeEvent<HTMLInputElement>) => {
        const { decide } = this.props;

        decide(event.target.value);
    };

    private _getEmailErrorMessage() {
        const { hasFocusEmail, isValidEmail, hasEmail, requiredFieldErrorMessage } = this.props;

        if (!isValidEmail && !hasFocusEmail) {
            return 'Введите корректный email';
        }

        if (!hasEmail) {
            return requiredFieldErrorMessage;
        }

        return '';
    }

    private _getAppealMessage() {
        const { appealDataFromDB } = this.props;

        if (!appealDataFromDB || !appealDataFromDB.verdict) {
            return '';
        }

        const { verdict, certId } = appealDataFromDB;

        return (
            <React.Fragment>
                <p>{alreadyAppealed}{appealTexts[verdict]}.</p>
                {
                    certId && (
                        <p>Был сформирован сертификат с номером {certId}.</p>
                    )
                }
            </React.Fragment>
        );
    }

    private readonly _confirm = () => {
        const {
            attemptId,
            verdict,
            email,
            confirm
        } = this.props;

        confirm(attemptId, verdict, email);
    };
}

export default VideosContent;
