import { handleActions } from 'redux-actions';

import { validateLogin } from 'client/store/common/helpers';

import {
    ActionTypes,
    BanFormPayload,
    IBanErrorAction,
    IBanFormState,
    IChooseExamsAction,
    ITypeLoginsAction,
    ITypeReasonAction
} from './types';

const initialState: IBanFormState = {
    examsForBan: [],
    logins: '',
    hasFocusLogins: false,
    isValidLogins: true,
    hasLogins: false,
    hasFocusReason: false,
    reason: '',
    isSuperban: false,
    isPending: false,
    requestError: null,
    isBanSuccess: false,
    isVisibleConfirmationModal: false,
    submitErrors: null
};

interface IValidateData {
    logins: string,
    reason: string,
    examsForBan: string[],
    isSuperban: boolean
}

export function parseLogins(logins: string): string[] {
    return logins
        .replace(/\s+/g, '')
        .split(',')
        .filter(login => login.length);
}

// eslint-disable-next-line
function validateBanData(validateData: IValidateData) {
    const { logins, reason, examsForBan, isSuperban } = validateData;
    const validationErrors = [];
    const parsedLogins = parseLogins(logins);
    const isLoginsValid = parsedLogins.length && parsedLogins.every(validateLogin);

    if (!isLoginsValid) {
        validationErrors.push('Введите корректные логины');
    }

    if (!reason.length) {
        validationErrors.push('Введите причину');
    }

    if (!isSuperban && !examsForBan.length) {
        validationErrors.push('Выберите экзамены');
    }

    return validationErrors;
}

function typeLogins(
    state: IBanFormState,
    action: ITypeLoginsAction
): IBanFormState {
    const { inputText } = action.payload;
    const logins = parseLogins(inputText);

    const isValidLogins = logins.every(validateLogin);

    return Object.assign({}, state, {
        logins: inputText,
        isValidLogins,
        hasLogins: inputText.length > 0,
        isBanSuccess: false,
        requestError: null,
        submitErrors: null
    });
}

function chooseExams(state: IBanFormState, action: IChooseExamsAction): IBanFormState {
    return Object.assign({}, state, {
        examsForBan: action.payload.exams,
        isBanSuccess: false,
        requestError: null,
        submitErrors: null
    });
}

function focusLogins(state: IBanFormState): IBanFormState {
    return Object.assign({}, state, {
        hasFocusLogins: true,
        isBanSuccess: false,
        requestError: null,
        submitErrors: null
    });
}

function blurLogins(state: IBanFormState): IBanFormState {
    return Object.assign({}, state, { hasFocusLogins: false });
}

function typeReason(
    state: IBanFormState,
    action: ITypeReasonAction
) : IBanFormState {
    const { inputText } = action.payload;

    return Object.assign({}, state, {
        reason: inputText,
        isBanSuccess: false,
        requestError: null,
        submitErrors: null
    });
}

function focusReason(state: IBanFormState): IBanFormState {
    return Object.assign({}, state, {
        hasFocusReason: true,
        isBanSuccess: false,
        requestError: null,
        submitErrors: null
    });
}

function blurReason(state: IBanFormState): IBanFormState {
    return Object.assign({}, state, { hasFocusReason: false });
}

function toggleSuperban(state: IBanFormState): IBanFormState {
    const { isSuperban } = state;

    return Object.assign({}, state, {
        isSuperban: !isSuperban,
        requestError: null,
        isBanSuccess: false,
        submitErrors: false
    });
}

function pressBan(state: IBanFormState): IBanFormState {
    const { logins, reason, examsForBan, isSuperban } = state;
    const validationErrors = validateBanData({
        logins,
        reason,
        examsForBan,
        isSuperban
    });

    return Object.assign({}, state, {
        isVisibleConfirmationModal: !validationErrors.length,
        submitErrors: validationErrors.length ? validationErrors : null
    });
}

function decline(state: IBanFormState): IBanFormState {
    return Object.assign({}, state, { isVisibleConfirmationModal: false });
}

function ban(state: IBanFormState): IBanFormState {
    return Object.assign({}, state, {
        isPending: true,
        submitErrors: null,
        isVisibleConfirmationModal: false,
        requestError: null
    });
}

function banSuccess(state: IBanFormState): IBanFormState {
    return Object.assign({}, state, {
        isPending: false,
        isBanSuccess: true,
        examsForBan: [],
        isSuperban: false
    });
}

function banError(state: IBanFormState, action: IBanErrorAction): IBanFormState {
    return Object.assign({}, state, { isPending: false, requestError: action.payload });
}

export default handleActions<IBanFormState, BanFormPayload>({
    [ActionTypes.CHOOSE_EXAMS]: chooseExams,
    [ActionTypes.TYPE_LOGINS]: typeLogins,
    [ActionTypes.FOCUS_LOGINS]: focusLogins,
    [ActionTypes.BLUR_LOGINS]: blurLogins,
    [ActionTypes.TYPE_REASON]: typeReason,
    [ActionTypes.FOCUS_REASON]: focusReason,
    [ActionTypes.BLUR_REASON]: blurReason,
    [ActionTypes.TOGGLE_SUPERBAN]: toggleSuperban,
    [ActionTypes.PRESS_BAN]: pressBan,
    [ActionTypes.DECLINE]: decline,
    [ActionTypes.BAN]: ban,
    [ActionTypes.BAN_SUCCESS]: banSuccess,
    [ActionTypes.BAN_ERROR]: banError
}, initialState);
