import forEach from 'lodash/forEach';
import map from 'lodash/map';
import reduce from 'lodash/reduce';
import some from 'lodash/some';
import get from 'lodash/get';

import ITestData, {
    IAnswer,
    ICategory,
    IQuestion,
    ISection,
    ITrialTemplateAllowedFails,
    ITrialTemplateToSections
} from 'common/types/testData';
import IEditTestData, {
    IEditCategories,
    IEditQuestions,
    IEditAnswers,
    TEditSection,
    TEditAnswer
} from 'common/types/editTestData';

export function getTrialTemplateId(exam: ITestData) {
    return exam.trialTemplateToSections[0].trialTemplateId;
}

function getActive(...removed: boolean[]): number {
    return removed.some(isRemoved => isRemoved) ? 0 : 1;
}

function getSectionAllowedFails(
    trialTemplateAllowedFails: ITrialTemplateAllowedFails[],
    code: string,
    trialTemplateId: number
) {
    const allowedFailsInfo = trialTemplateAllowedFails.find(item => {
        return item.sectionId.code === code && item.trialTemplateId === trialTemplateId;
    });

    return allowedFailsInfo!.allowedFails;
}

function getCategoriesIds(
    trialTemplateToSections: ITrialTemplateToSections[],
    sectionCode: string
): number[] {
    return trialTemplateToSections
        .filter(item => {
            return item.sectionId.code === sectionCode;
        })
        .map(item => item.categoryId.id);
}

interface IPrepareAnswersParams {
    exam: ITestData,
    sectionCode: string,
    categoryId: number,
    question: IQuestion,
    maxIds: IMaxIds
}

export function escapeHtml(text: string) {
    return text.toString()
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;');
}

function replaceBRAndAmp(text: string) {
    return text
        .replace(/&lt;br\/?&gt;/g, '<br>')
        .replace(/&amp;/g, '&');
}

function prepareAnswers(params: IPrepareAnswersParams): IEditAnswers {
    const { exam, sectionCode, categoryId, question, maxIds } = params;
    const { answers: examAnswers } = exam;
    const { id: questionId, text: questionText } = question;
    const answers = {} as IEditAnswers;

    examAnswers
        .filter(({ questionId: examQuestion }: IAnswer) => {
            return examQuestion.sectionId.code === sectionCode &&
                examQuestion.categoryId.id === categoryId &&
                examQuestion.id === questionId
                && examQuestion.text === questionText;
        })
        .map(answer => {
            const { text, correct, active } = answer;
            let { id } = answer;
            let isNew = false;

            if (id === 0) {
                id = createId(maxIds, 'answer');
                isNew = true;
            }

            answers[id] = {
                isNew,
                isChanged: false,
                isRemoved: !active,
                id,
                text: escapeHtml(text),
                correct,
                active
            };
        });

    return answers;
}

interface IPrepareQuestionsParams {
    exam: ITestData,
    sectionCode: string,
    categoryId: number,
    maxIds: IMaxIds
}

function prepareQuestions(params: IPrepareQuestionsParams) {
    const { exam, sectionCode, categoryId: currentCategoryId, maxIds } = params;
    const { questions: examQuestions } = exam;
    const questions = {} as IEditQuestions;

    examQuestions
        .filter(({ sectionId, categoryId }: IQuestion) => {
            return sectionId.code === sectionCode && categoryId.id === currentCategoryId;
        })
        .map(question => {
            const { text, type, active } = question;
            let { id } = question;
            let isNew = false;

            if (id === 0) {
                id = createId(maxIds, 'question');
                isNew = true;
            }

            const answers = prepareAnswers({
                exam,
                sectionCode,
                categoryId: currentCategoryId,
                question,
                maxIds
            });

            questions[id] = {
                isNew,
                isChanged: false,
                isRemoved: !active,
                id,
                type,
                text: escapeHtml(text),
                answers,
                active
            };
        });

    return questions;
}

function getQuantity(
    exam: ITestData,
    sectionCode: string,
    categoryId: number
) {
    const { trialTemplateToSections } = exam;
    const trialTemplateToSection = trialTemplateToSections.find(
        ({ categoryId: category, sectionId: section }) => {
            return categoryId === category.id && sectionCode === section.code;
        }
    );

    return trialTemplateToSection!.quantity;
}

interface IMaxIds {
    category: number,
    question: number,
    answer: number
}

interface IPrepareCategoriesParams {
    exam: ITestData,
    sectionCode: string,
    categoryIds: number[],
    maxIds: IMaxIds
}

function prepareCategories(params: IPrepareCategoriesParams): IEditCategories {
    const { exam, sectionCode, categoryIds, maxIds } = params;
    const categories = {} as IEditCategories;
    const { trialTemplateToSections } = exam;

    categoryIds.map(id => {
        const toSectionAndCategory = trialTemplateToSections.find(data => {
            return data.sectionId.code === sectionCode && data.categoryId.id === id;
        });

        categories[id] = {
            isNew: false,
            isChanged: false,
            isRemoved: toSectionAndCategory!.quantity === 0,
            id,
            quantity: getQuantity(exam, sectionCode, id),
            questions: prepareQuestions({
                exam,
                maxIds,
                sectionCode,
                categoryId: id
            })
        };
    });

    return categories;
}

interface IWithId {
    id: number;
}

export function getMaxIdFromExam<T extends IWithId>(array: T[]): number {
    let max = 0;

    array.forEach((item: T) => {
        if (item.id > max) {
            max = item.id;
        }
    });

    return max;
}

type IMaxIdField = 'question' | 'answer' | 'category';

export function createId(maxIds: IMaxIds, name: IMaxIdField) {
    maxIds[name] += 1;

    return maxIds[name];
}

function getSectionsFromEditData(editData: IEditTestData): ISection[] {
    return reduce(editData, (res: ISection[], section: TEditSection) => {
        const { serviceId, code, title } = section;

        res.push({
            serviceId,
            code,
            title
        });

        return res;
    }, []);
}

function getCategoriesFromEditData(editData: IEditTestData): ICategory[] {
    return reduce(editData, (res: ICategory[], section: TEditSection) => {
        const { categories } = section;

        map(categories, category => {
            const { id } = category;

            res.push({
                id,
                difficulty: 0,
                timeLimit: 0
            });
        });

        return res;
    }, []);
}

function getTrialTemplateAllowedFails(
    editData: IEditTestData,
    trialTemplateId: number
): ITrialTemplateAllowedFails[] {
    return reduce(editData, (res: ITrialTemplateAllowedFails[], section: TEditSection) => {
        const { serviceId, code, title, allowedFails } = section;

        res.push({
            sectionId: {
                serviceId,
                code,
                title
            },
            allowedFails,
            trialTemplateId
        });

        return res;
    }, []);
}

function getTrialTemplateToSections(
    editData: IEditTestData,
    trialTemplateId: number
): ITrialTemplateToSections[] {
    return reduce(editData, (res: ITrialTemplateToSections[], section: TEditSection) => {
        const { categories, title, code, serviceId, isRemoved: isSectionRemoved } = section;

        forEach(categories, category => {
            const { id, quantity, questions, isRemoved: isCategoryRemoved } = category;
            const activeQuestionsCount = countActiveItems(questions);
            const isRemoved = isSectionRemoved || isCategoryRemoved || activeQuestionsCount === 0;

            res.push({
                trialTemplateId,
                quantity: isRemoved ? 0 : quantity,
                sectionId: {
                    serviceId,
                    code,
                    title
                },
                categoryId: {
                    id,
                    difficulty: 0,
                    timeLimit: 0
                }
            });
        });

        return res;
    }, []);
}

function getQuestionsFromEditData(editData: IEditTestData): IQuestion[] {
    return reduce(editData, (res: IQuestion[], section: TEditSection) => {
        const { serviceId, code, title, categories, isRemoved: isSectionRemoved } = section;

        forEach(categories, category => {
            const { id: categoryId, questions, isRemoved: isCategoryRemoved } = category;

            forEach(questions, question => {
                const {
                    isRemoved: isQuestionRemoved,
                    isNew,
                    text,
                    type,
                    id: questionId
                } = question;

                res.push({
                    sectionId: {
                        serviceId,
                        code,
                        title
                    },
                    categoryId: {
                        id: categoryId,
                        difficulty: 0,
                        timeLimit: 0
                    },
                    id: isNew ? 0 : questionId,
                    active: getActive(isSectionRemoved, isCategoryRemoved, isQuestionRemoved),
                    text: replaceBRAndAmp(text),
                    type
                });
            });
        });

        return res;
    }, []);
}

function getAnswersFromEditData(editData: IEditTestData): IAnswer[] {
    return reduce(editData, (res: IAnswer[], section: TEditSection) => {
        const { serviceId, code, title, categories, isRemoved: isSectionRemoved } = section;

        forEach(categories, category => {
            const { id: categoryId, questions, isRemoved: isCategoryRemoved } = category;

            forEach(questions, question => {
                const {
                    text: questionText,
                    type,
                    isRemoved: isQuestionRemoved,
                    id: questionId,
                    answers,
                    isNew: isNewQuestion
                } = question;

                // eslint-disable-next-line max-nested-callbacks
                forEach(answers, answer => {
                    const {
                        isRemoved: isAnswerRemoved,
                        isNew,
                        text,
                        correct,
                        id: answerId
                    } = answer;

                    res.push({
                        questionId: {
                            sectionId: {
                                serviceId,
                                code,
                                title
                            },
                            categoryId: {
                                id: categoryId,
                                difficulty: 0,
                                timeLimit: 0
                            },
                            id: isNewQuestion ? 0 : questionId,
                            active: getActive(
                                isSectionRemoved, isCategoryRemoved, isQuestionRemoved
                            ),
                            text: replaceBRAndAmp(questionText),
                            type
                        },
                        id: isNew ? 0 : answerId,
                        correct,
                        text: replaceBRAndAmp(text),
                        active: getActive(
                            isSectionRemoved, isCategoryRemoved, isQuestionRemoved, isAnswerRemoved
                        )
                    });
                });
            });
        });

        return res;
    }, []);
}

function removeUnusedData(editData: IEditTestData): void {
    forEach(editData, (section, sectionCode) => {
        if (section.isNew && section.isRemoved) {
            delete editData[sectionCode];

            return;
        }

        forEach(section.categories, (category, categoryId) => {
            if (category.isNew && category.isRemoved) {
                delete section.categories[Number(categoryId)];

                return;
            }

            forEach(category.questions, (question, questionId) => {
                if (question.isNew && question.isRemoved) {
                    delete category.questions[Number(questionId)];

                    return;
                }

                // eslint-disable-next-line max-nested-callbacks
                forEach(question.answers, (answer, answerId) => {
                    if (answer.isNew && answer.isRemoved) {
                        delete question.answers[Number(answerId)];
                    }
                });
            });
        });
    });
}

export function getCorrectAnswersCount(answers: IEditAnswers) {
    return reduce(answers, (count: number, answer: TEditAnswer) => {
        if (!answer.isRemoved) {
            count += answer.correct;
        }

        return count;
    }, 0);
}

export function countActiveItems(items: any): number {
    return reduce(items, (count: number, item) => {
        if (!item.isRemoved) {
            count += 1;
        }

        return count;
    }, 0);
}

export interface IValidateQuestionsParams {
    editData: IEditTestData,
    sectionCode: string,
    categoryId: number
}

export function validateQuestions(data: IValidateQuestionsParams): string[] {
    const validationErrors = [] as string[];
    const { sectionCode, categoryId, editData } = data;
    const category = editData[sectionCode].categories[categoryId];
    const questionsIds = Object.keys(category.questions);
    const activeQuestionsCount = countActiveItems(category.questions);

    if (activeQuestionsCount > 0 && category.quantity === 0) {
        validationErrors.push('Не указано количество вопросов в категории');
    }

    // eslint-disable-next-line complexity,max-statements
    questionsIds.forEach(questionId => {
        const question = category.questions[Number(questionId)];
        const { text, type, answers, isRemoved } = question;
        const answersIds = Object.keys(answers);

        if (isRemoved) {
            return;
        }

        if (!text) {
            validationErrors.push('У одного из вопросов нет текста');
        }

        const activeAnswersCount = countActiveItems(answers);

        if (activeAnswersCount === 0) {
            validationErrors.push(`У вопроса "${text}" нет ответов`);

            return;
        }

        const correctAnswersCount = getCorrectAnswersCount(answers);

        if (correctAnswersCount === 0) {
            validationErrors.push(`У вопроса "${text}" нет правильного ответа`);
        }

        if (type === 0 && correctAnswersCount > 1) {
            validationErrors.push(`У вопроса "${text}" больше одного правильного ответа`);
        }

        answersIds.forEach(answerId => {
            const { isRemoved: isAnswerRemoved, text: answerText } = answers[Number(answerId)];

            if (isAnswerRemoved) {
                return;
            }

            if (!answerText) {
                validationErrors.push(`У ответа к вопросу "${text}" нет текста`);
            }
        });
    });

    return validationErrors;
}

export function validateSections(editData: IEditTestData): string[] {
    const validationErrors = [] as string[];

    // eslint-disable-next-line max-statements
    forEach(editData, (section, sectionCode) => {
        const { categories, isRemoved: isSectionRemoved } = section;

        if (isSectionRemoved) {
            return;
        }

        const categoriesIds = Object.keys(categories);

        if (categoriesIds.length === 0) {
            validationErrors.push(`В секции ${sectionCode} нет категорий`);

            return;
        }

        const activeCategoriesCount = countActiveItems(categories);

        if (activeCategoriesCount === 0) {
            validationErrors.push(`В секции ${sectionCode} нет категорий`);

            return;
        }

        // eslint-disable-next-line max-statements
        categoriesIds.forEach(categoryId => {
            const category = categories[Number(categoryId)];
            const { isRemoved: isCategoryRemoved, questions } = category;

            if (isCategoryRemoved) {
                return;
            }

            const questionsIds = Object.keys(questions);

            if (questionsIds.length === 0) {
                validationErrors.push(`В категории ${categoryId} нет вопросов`);

                return;
            }

            const activeQuestionsCount = countActiveItems(questions);

            if (activeQuestionsCount === 0) {
                validationErrors.push(`В категории ${categoryId} нет вопросов`);
            }
        });
    });

    return validationErrors;
}

export function itemsHasChanges(items: any): boolean {
    const itemsKeys = Object.keys(items) || [];

    // eslint-disable-next-line complexity
    return itemsKeys.some(itemKey => {
        const item = items[itemKey] || {};
        const { isNew, isChanged, isRemoved } = item;
        const newRemoved = isNew && isRemoved;

        return !newRemoved && (isNew || isChanged || isRemoved);
    });
}

// eslint-disable-next-line complexity
export function questionsHasChanges(questions: IEditQuestions): boolean {
    const questionsIds = Object.keys(questions);

    for (const questionId of questionsIds) {
        const { isNew, isChanged, isRemoved, answers } = questions[Number(questionId)];
        const newRemoved = isNew && isRemoved;

        if (!newRemoved && (isNew || isChanged || isRemoved || itemsHasChanges(answers))) {
            return true;
        }
    }

    return false;
}

// eslint-disable-next-line complexity
export function categoriesHasChanges(categories: IEditCategories): boolean {
    const categoriesIds = Object.keys(categories);

    for (const categoryId of categoriesIds) {
        const { isNew, isChanged, isRemoved, questions } = categories[Number(categoryId)];
        const newRemoved = isNew && isRemoved;

        if (!newRemoved && (isNew || isChanged || isRemoved || questionsHasChanges(questions))) {
            return true;
        }
    }

    return false;
}

export function examToEditData(exam: ITestData) {
    const editData = {} as IEditTestData;
    const maxIds = {
        category: getMaxIdFromExam(exam.categories),
        question: getMaxIdFromExam(exam.questions),
        answer: getMaxIdFromExam(exam.answers)
    };

    exam.sections.map(section => {
        const { trialTemplateAllowedFails, trialTemplateToSections } = exam;
        const { title, code, serviceId } = section;

        const trialTemplateId = getTrialTemplateId(exam);

        const allowedFails = getSectionAllowedFails(
            trialTemplateAllowedFails,
            code,
            trialTemplateId
        );

        const examCategories = getCategoriesIds(trialTemplateToSections, section.code);

        editData[section.code] = {
            isNew: false,
            isChanged: false,
            isRemoved: false,
            code,
            title,
            allowedFails,
            serviceId,
            categories: prepareCategories({
                exam,
                sectionCode: code,
                maxIds,
                categoryIds: examCategories
            })
        };

        const activeCategoriesCount = countActiveItems(editData[section.code].categories);

        editData[section.code].isRemoved = activeCategoriesCount === 0;
    });

    return {
        maxIds,
        editData
    };
}

export function setIsChanged(draftData: IEditTestData, publishedData: IEditTestData): void {
    forEach(draftData, (section, sectionCode) => {
        const publishedSection = publishedData[sectionCode];

        forEach(section.categories, (category, categoryId) => {
            const publishedCategory = get(publishedSection, `categories[${categoryId}]`, {});

            // eslint-disable-next-line complexity
            forEach(category.questions, (question, questionId) => {
                const publishedQuestion = get(publishedCategory, `questions[${questionId}]`, {});

                // eslint-disable-next-line max-nested-callbacks
                forEach(question.answers, (answer, answerId) => {
                    const publishedAnswer = get(publishedQuestion, `answers[${answerId}]`, {});
                    const isAnswerChanged = !publishedAnswer ||
                        answer.text !== publishedAnswer.text ||
                        answer.correct !== publishedAnswer.correct ||
                        answer.active !== publishedAnswer.active;

                    if (isAnswerChanged) {
                        answer.isChanged = true;
                    }
                });

                const isAnswersChanged = some(question.answers, 'isChanged');
                const isQuestionChanged = question.active !== publishedQuestion.active ||
                    question.text !== publishedQuestion.text ||
                    question.type !== publishedQuestion.type;

                if (!publishedQuestion || isQuestionChanged || isAnswersChanged) {
                    question.isChanged = true;
                }
            });
        });
    });
}

export function editDataToExam(editData: IEditTestData, trialTemplateId: number): ITestData {
    removeUnusedData(editData);

    return {
        sections: getSectionsFromEditData(editData),
        categories: getCategoriesFromEditData(editData),
        trialTemplateAllowedFails: getTrialTemplateAllowedFails(editData, trialTemplateId),
        trialTemplateToSections: getTrialTemplateToSections(editData, trialTemplateId),
        questions: getQuestionsFromEditData(editData),
        answers: getAnswersFromEditData(editData)
    };
}
