import cloneDeep from 'lodash/cloneDeep';
import get from 'lodash/get';

import { handleActions } from 'redux-actions';

import RequestStatus from 'common/types/requestStatus';
import DraftStatus from 'common/types/draftStatus';

import { examToEditData, setIsChanged } from './helpers';

import {
    ActionTypes,
    IAddAnswer,
    IAddCategory,
    IAddQuestion,
    IAddSection,
    ICancelChangesError,
    IChangeAnswerText,
    IChangeQuestionText,
    IDeleteDraftError,
    IDownloadJSONError,
    IDownloadJSONSuccess,
    IEditAnswerCorrect,
    IEditCategoryQuantity,
    IEditQuestionType,
    IEditSectionAllowedFails,
    IExamDataPayload,
    IExamDataState,
    ILoadFileError,
    ILoadFileSuccess,
    IOpenCategory,
    IPublishFailed,
    ISaveDraftError,
    ISendDraftForModerationError,
    IStartEditError,
    IStartEditSuccess,
    IToggleRemoveAnswer,
    IToggleRemoveCategory,
    IToggleRemoveQuestion,
    IToggleRemoveSection
} from './types';

export const initialState: IExamDataState = {
    draftStatus: null,
    loadStatus: null,
    startEditStatus: null,
    requestError: null,
    exam: null,
    editData: null,
    maxIds: null,
    openedCategoryId: null,
    openedSectionCode: null,
    publishStatus: null,
    deleteDraftStatus: null,
    isVisiblePublishModal: false,
    isVisibleDeleteDraftModal: false,
    isVisibleCancelChangesModal: false,
    cancelChangesStatus: null,
    isEditMode: false,
    saveDraftStatus: null,
    sendDraftForModerationStatus: null,
    notEditedData: null
};

function loadFile(state: IExamDataState): IExamDataState {
    return {
        ...state,
        loadStatus: RequestStatus.pending,
        requestError: null,
        saveDraftStatus: null,
        publishStatus: null,
        sendDraftForModerationStatus: null,
        deleteDraftStatus: null,
        exam: null,
        editData: null,
        notEditedData: null,
        maxIds: null
    };
}

function loadFileSuccess(state: IExamDataState, action: ILoadFileSuccess): IExamDataState {
    const { fileData, publishedData } = action.payload;
    const { editData, maxIds } = examToEditData(fileData);
    const { editData: notEditedData } = examToEditData(publishedData);

    setIsChanged(editData, notEditedData);

    return {
        ...state,
        loadStatus: RequestStatus.success,
        exam: fileData,
        editData,
        notEditedData,
        maxIds,
        isEditMode: true
    };
}

function loadFileError(state: IExamDataState, action: ILoadFileError): IExamDataState {
    return {
        ...state,
        loadStatus: RequestStatus.failed,
        requestError: action.payload
    };
}

function downloadJSON(state: IExamDataState): IExamDataState {
    return {
        ...state,
        loadStatus: RequestStatus.pending,
        publishStatus: null,
        saveDraftStatus: null,
        sendDraftForModerationStatus: null,
        deleteDraftStatus: null,
        requestError: null,
        exam: null,
        maxIds: null,
        editData: null,
        notEditedData: null
    };
}

function downloadJSONSuccess(state: IExamDataState, action: IDownloadJSONSuccess): IExamDataState {
    const { publishedData } = action.payload;
    const { editData, maxIds } = examToEditData(publishedData);
    const notEditedData = cloneDeep(editData);

    return {
        ...state,
        loadStatus: RequestStatus.success,
        exam: publishedData,
        editData,
        notEditedData,
        maxIds
    };
}

function downloadJSONError(state: IExamDataState, action: IDownloadJSONError): IExamDataState {
    return {
        ...state,
        loadStatus: RequestStatus.failed,
        requestError: action.payload
    };
}

function openCategory(state: IExamDataState, action: IOpenCategory): IExamDataState {
    return {
        ...state,
        openedSectionCode: action.payload.sectionCode,
        openedCategoryId: action.payload.categoryId,
        requestError: null
    };
}

function closeCategory(state: IExamDataState): IExamDataState {
    return {
        ...state,
        openedSectionCode: null,
        openedCategoryId: null
    };
}

function publishStarted(state: IExamDataState): IExamDataState {
    return {
        ...state,
        publishStatus: RequestStatus.pending,
        loadStatus: null,
        saveDraftStatus: null,
        sendDraftForModerationStatus: null,
        deleteDraftStatus: null,
        requestError: null,
        isVisiblePublishModal: false
    };
}

function publishSuccess(state: IExamDataState): IExamDataState {
    return {
        ...state,
        exam: null,
        editData: null,
        notEditedData: null,
        loadStatus: null,
        publishStatus: RequestStatus.success,
        requestError: null,
        draftStatus: DraftStatus.published,
        isEditMode: false
    };
}

function publishFailed(state: IExamDataState, action: IPublishFailed): IExamDataState {
    return {
        ...state,
        publishStatus: RequestStatus.failed,
        requestError: action.payload
    };
}

function declinePublish(state: IExamDataState): IExamDataState {
    return {
        ...state,
        isVisiblePublishModal: false
    };
}

function showPublishModal(state: IExamDataState): IExamDataState {
    return {
        ...state,
        isVisiblePublishModal: true
    };
}

function deleteDraft(state: IExamDataState): IExamDataState {
    return {
        ...state,
        isVisibleDeleteDraftModal: false,
        deleteDraftStatus: RequestStatus.pending,
        publishStatus: null,
        saveDraftStatus: null,
        sendDraftForModerationStatus: null,
        requestError: null
    };
}

function deleteDraftSuccess(state: IExamDataState): IExamDataState {
    return {
        ...state,
        deleteDraftStatus: RequestStatus.success,
        publishStatus: null,
        saveDraftStatus: null,
        exam: null,
        editData: null,
        notEditedData: null,
        isEditMode: false,
        draftStatus: DraftStatus.ignored
    };
}

function deleteDraftError(state: IExamDataState, action: IDeleteDraftError): IExamDataState {
    return {
        ...state,
        deleteDraftStatus: RequestStatus.failed,
        requestError: action.payload
    };
}

function pressDeleteDraft(state: IExamDataState): IExamDataState {
    return {
        ...state,
        isVisibleDeleteDraftModal: true
    };
}

function declineDeleteDraft(state: IExamDataState): IExamDataState {
    return {
        ...state,
        isVisibleDeleteDraftModal: false
    };
}

function startEdit(state: IExamDataState): IExamDataState {
    return {
        ...state,
        requestError: null,
        startEditStatus: RequestStatus.pending,
        publishStatus: null,
        saveDraftStatus: null,
        sendDraftForModerationStatus: null,
        deleteDraftStatus: null,
        exam: null,
        editData: null,
        notEditedData: null,
        maxIds: null
    };
}

function startEditError(state: IExamDataState, action: IStartEditError): IExamDataState {
    return {
        ...state,
        requestError: action.payload,
        isEditMode: false,
        startEditStatus: RequestStatus.failed
    };
}

function startEditSuccess(state: IExamDataState, action: IStartEditSuccess): IExamDataState {
    const { draft, publishedData } = action.payload;
    const { editData, maxIds } = examToEditData(draft);
    const { editData: notEditedData } = examToEditData(publishedData);

    setIsChanged(editData, notEditedData);

    return {
        ...state,
        requestError: null,
        isEditMode: true,
        startEditStatus: RequestStatus.success,
        exam: draft,
        editData,
        notEditedData,
        maxIds
    };
}

function saveDraft(state: IExamDataState): IExamDataState {
    return {
        ...state,
        requestError: null,
        saveDraftStatus: RequestStatus.pending,
        sendDraftForModerationStatus: null,
        publishStatus: null,
        startEditStatus: null,
        deleteDraftStatus: null
    };
}

function saveDraftError(state: IExamDataState, action: ISaveDraftError): IExamDataState {
    return {
        ...state,
        saveDraftStatus: RequestStatus.failed,
        requestError: action.payload
    };
}

function saveDraftSuccess(state: IExamDataState): IExamDataState {
    return {
        ...state,
        saveDraftStatus: RequestStatus.success,
        isEditMode: false,
        exam: null,
        editData: null,
        notEditedData: null,
        draftStatus: DraftStatus.saved
    };
}

function sendDraftForModeration(state: IExamDataState): IExamDataState {
    return {
        ...state,
        saveDraftStatus: null,
        publishStatus: null,
        deleteDraftStatus: null,
        startEditStatus: null,
        sendDraftForModerationStatus: RequestStatus.pending,
        requestError: null
    };
}

function sendDraftForModerationError(
    state: IExamDataState,
    action: ISendDraftForModerationError
): IExamDataState {
    return {
        ...state,
        sendDraftForModerationStatus: RequestStatus.failed,
        requestError: action.payload
    };
}

function sendDraftForModerationSuccess(state: IExamDataState): IExamDataState {
    return {
        ...state,
        sendDraftForModerationStatus: RequestStatus.success,
        isEditMode: false,
        exam: null,
        editData: null,
        notEditedData: null,
        draftStatus: DraftStatus.onModeration
    };
}

function editAnswerCorrect(state: IExamDataState, action: IEditAnswerCorrect): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, categoryId, questionId, answerId, correct } = action.payload;
    const answer = newEditData![sectionCode].categories[categoryId]
        .questions[questionId].answers[answerId];
    const oldAnswer = get(
        state,
        // eslint-disable-next-line max-len
        `notEditedData[${sectionCode}].categories[${categoryId}].questions[${questionId}].answers[${answerId}]`,
        {}
    );

    answer.correct = correct;
    answer.isChanged = answer.text !== oldAnswer.text || answer.correct !== oldAnswer.correct;

    return {
        ...state,
        editData: newEditData
    };
}

function editQuestionType(state: IExamDataState, action: IEditQuestionType): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, categoryId, questionId, type } = action.payload;
    const question = newEditData![sectionCode].categories[categoryId].questions[questionId];
    const oldQuestion = get(
        state,
        `notEditedData[${sectionCode}].categories[${categoryId}].questions[${questionId}]`,
        {}
    );

    question.type = type;
    question.isChanged = question.text !== oldQuestion.text || question.type !== oldQuestion.type;

    return {
        ...state,
        editData: newEditData
    };
}

function editCategoryQuantity(
    state: IExamDataState,
    action: IEditCategoryQuantity
): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, categoryId, quantity } = action.payload;
    const category = newEditData![sectionCode].categories[categoryId];
    const oldCategory = get(state, `notEditedData[${sectionCode}].categories[${categoryId}]`, {});

    category.quantity = quantity;
    category.isChanged = category.quantity !== oldCategory.quantity;

    return {
        ...state,
        editData: newEditData
    };
}

function editSectionAllowedFails(
    state: IExamDataState,
    action: IEditSectionAllowedFails
): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, allowedFails } = action.payload;
    const section = newEditData![sectionCode];
    const oldSection = get(state, `notEditedData[${sectionCode}]`, {});

    section.allowedFails = allowedFails;
    section.isChanged = section.allowedFails !== oldSection.allowedFails;

    return {
        ...state,
        editData: newEditData,
        requestError: null
    };
}

function toggleRemoveAnswer(
    state: IExamDataState,
    action: IToggleRemoveAnswer
): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, categoryId, questionId, answerId, isRemoved } = action.payload;
    const answer = newEditData![sectionCode].categories[categoryId]
        .questions[questionId].answers[answerId];

    answer.isRemoved = isRemoved;

    return {
        ...state,
        editData: newEditData
    };
}

function toggleRemoveQuestion(
    state: IExamDataState,
    action: IToggleRemoveQuestion
): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, categoryId, questionId, isRemoved } = action.payload;
    const question = newEditData![sectionCode].categories[categoryId].questions[questionId];

    question.isRemoved = isRemoved;

    return {
        ...state,
        editData: newEditData
    };
}

function toggleRemoveCategory(
    state: IExamDataState,
    action: IToggleRemoveCategory
): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, categoryId, isRemoved } = action.payload;
    const category = newEditData![sectionCode].categories[categoryId];

    category.isRemoved = isRemoved;

    return {
        ...state,
        editData: newEditData,
        requestError: null
    };
}

function toggleRemoveSection(
    state: IExamDataState,
    action: IToggleRemoveSection
): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, isRemoved } = action.payload;
    const section = newEditData![sectionCode];

    section.isRemoved = isRemoved;

    return {
        ...state,
        editData: newEditData,
        requestError: null
    };
}

function changeAnswerText(state: IExamDataState, action: IChangeAnswerText): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, categoryId, questionId, answerId, text } = action.payload;
    const answer = newEditData![sectionCode].categories[categoryId]
        .questions[questionId].answers[answerId];
    const oldAnswer = get(
        state,
        // eslint-disable-next-line max-len
        `notEditedData[${sectionCode}].categories[${categoryId}].questions[${questionId}].answers[${answerId}]`,
        {}
    );

    answer.text = text;
    answer.isChanged = answer.text !== oldAnswer.text || answer.correct !== oldAnswer.correct;

    return {
        ...state,
        editData: newEditData
    };
}

function changeQuestionText(state: IExamDataState, action: IChangeQuestionText): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const { sectionCode, categoryId, questionId, text } = action.payload;
    const question = newEditData![sectionCode].categories[categoryId].questions[questionId];
    const oldQuestion = get(
        state,
        `notEditedData[${sectionCode}].categories[${categoryId}].questions[${questionId}]`,
        {}
    );

    question.text = text;
    question.isChanged = question.text !== oldQuestion.text || question.type !== oldQuestion.type;

    return {
        ...state,
        editData: newEditData
    };
}

function addAnswer(state: IExamDataState, action: IAddAnswer): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const newMaxIds = cloneDeep(state.maxIds);
    const { sectionCode, categoryId, questionId } = action.payload;

    newMaxIds!.answer += 1;

    const answerId = newMaxIds!.answer;

    newEditData![sectionCode].categories[categoryId]
        .questions[questionId].answers[answerId] = {
            isNew: true,
            isChanged: false,
            isRemoved: false,
            id: answerId,
            text: '',
            correct: 0,
            active: 1
        };

    return {
        ...state,
        editData: newEditData,
        maxIds: newMaxIds
    };
}

function addQuestion(state: IExamDataState, action: IAddQuestion): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const newMaxIds = cloneDeep(state.maxIds);
    const { sectionCode, categoryId } = action.payload;

    newMaxIds!.question += 1;

    const questionId = newMaxIds!.question;

    newEditData![sectionCode].categories[categoryId].questions[questionId] = {
        isNew: true,
        isChanged: false,
        isRemoved: false,
        id: questionId,
        text: '',
        type: 0,
        active: 1,
        answers: {}
    };

    return {
        ...state,
        editData: newEditData,
        maxIds: newMaxIds
    };
}

function addCategory(state: IExamDataState, action: IAddCategory): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const newMaxIds = cloneDeep(state.maxIds);
    const { sectionCode } = action.payload;

    newMaxIds!.category += 1;

    const categoryId = newMaxIds!.category;

    newEditData![sectionCode].categories[categoryId] = {
        isNew: true,
        isChanged: false,
        isRemoved: false,
        id: categoryId,
        quantity: 0,
        questions: {}
    };

    return {
        ...state,
        editData: newEditData,
        maxIds: newMaxIds,
        requestError: null
    };
}

function addSection(state: IExamDataState, action: IAddSection): IExamDataState {
    const newEditData = cloneDeep(state.editData);
    const newMaxIds = cloneDeep(state.maxIds);
    const { sectionCode, serviceId, title, allowedFails } = action.payload;

    newMaxIds!.category += 1;

    const categoryId = newMaxIds!.category;

    newEditData![sectionCode] = {
        isNew: true,
        isChanged: false,
        isRemoved: false,
        code: sectionCode,
        serviceId,
        title,
        allowedFails,
        categories: {
            [categoryId]: {
                isNew: true,
                isChanged: false,
                isRemoved: false,
                id: categoryId,
                quantity: 0,
                questions: {}
            }
        }
    };

    return {
        ...state,
        editData: newEditData,
        maxIds: newMaxIds,
        requestError: null
    };
}

function cancelChanges(state: IExamDataState): IExamDataState {
    return {
        ...state,
        cancelChangesStatus: RequestStatus.pending,
        isVisibleCancelChangesModal: false,
        requestError: null
    };
}

function cancelChangesSuccess(state: IExamDataState): IExamDataState {
    return {
        ...state,
        cancelChangesStatus: RequestStatus.success,
        isEditMode: false,
        exam: null,
        editData: null,
        notEditedData: null,
        maxIds: null
    };
}

function cancelChangesError(state: IExamDataState, action: ICancelChangesError): IExamDataState {
    return {
        ...state,
        cancelChangesStatus: RequestStatus.failed,
        requestError: action.payload
    };
}

function showCancelChangesModal(state: IExamDataState): IExamDataState {
    return {
        ...state,
        isVisibleCancelChangesModal: true
    };
}

function declineCancelChanges(state: IExamDataState): IExamDataState {
    return {
        ...state,
        isVisibleCancelChangesModal: false
    };
}

export default handleActions<IExamDataState, IExamDataPayload>({
    [ActionTypes.LOAD_FILE]: loadFile,
    [ActionTypes.LOAD_FILE_SUCCESS]: loadFileSuccess,
    [ActionTypes.LOAD_FILE_ERROR]: loadFileError,
    [ActionTypes.OPEN_CATEGORY]: openCategory,
    [ActionTypes.CLOSE_CATEGORY]: closeCategory,
    [ActionTypes.DOWNLOAD_JSON]: downloadJSON,
    [ActionTypes.DOWNLOAD_JSON_SUCCESS]: downloadJSONSuccess,
    [ActionTypes.DOWNLOAD_JSON_ERROR]: downloadJSONError,
    [ActionTypes.PUBLISH_EXAM]: publishStarted,
    [ActionTypes.PUBLISH_SUCCESS]: publishSuccess,
    [ActionTypes.PUBLISH_FAILED]: publishFailed,
    [ActionTypes.DECLINE_PUBLISH]: declinePublish,
    [ActionTypes.PRESS_PUBLISH]: showPublishModal,
    [ActionTypes.DELETE_DRAFT]: deleteDraft,
    [ActionTypes.DELETE_DRAFT_SUCCESS]: deleteDraftSuccess,
    [ActionTypes.DELETE_DRAFT_ERROR]: deleteDraftError,
    [ActionTypes.PRESS_DELETE_DRAFT]: pressDeleteDraft,
    [ActionTypes.DECLINE_DELETE_DRAFT]: declineDeleteDraft,
    [ActionTypes.START_EDIT]: startEdit,
    [ActionTypes.START_EDIT_ERROR]: startEditError,
    [ActionTypes.START_EDIT_SUCCESS]: startEditSuccess,
    [ActionTypes.SAVE_DRAFT]: saveDraft,
    [ActionTypes.SAVE_DRAFT_ERROR]: saveDraftError,
    [ActionTypes.SAVE_DRAFT_SUCCESS]: saveDraftSuccess,
    [ActionTypes.SEND_DRAFT_FOR_MODERATION]: sendDraftForModeration,
    [ActionTypes.SEND_DRAFT_FOR_MODERATION_ERROR]: sendDraftForModerationError,
    [ActionTypes.SEND_DRAFT_FOR_MODERATION_SUCCESS]: sendDraftForModerationSuccess,
    [ActionTypes.EDIT_ANSWER_CORRECT]: editAnswerCorrect,
    [ActionTypes.EDIT_QUESTION_TYPE]: editQuestionType,
    [ActionTypes.EDIT_CATEGORY_QUANTITY]: editCategoryQuantity,
    [ActionTypes.EDIT_SECTION_ALLOWED_FAILS]: editSectionAllowedFails,
    [ActionTypes.TOGGLE_REMOVE_ANSWER]: toggleRemoveAnswer,
    [ActionTypes.TOGGLE_REMOVE_QUESTION]: toggleRemoveQuestion,
    [ActionTypes.TOGGLE_REMOVE_CATEGORY]: toggleRemoveCategory,
    [ActionTypes.TOGGLE_REMOVE_SECTION]: toggleRemoveSection,
    [ActionTypes.CHANGE_ANSWER_TEXT]: changeAnswerText,
    [ActionTypes.CHANGE_QUESTION_TEXT]: changeQuestionText,
    [ActionTypes.ADD_ANSWER]: addAnswer,
    [ActionTypes.ADD_QUESTION]: addQuestion,
    [ActionTypes.ADD_CATEGORY]: addCategory,
    [ActionTypes.ADD_SECTION]: addSection,
    [ActionTypes.CANCEL_CHANGES]: cancelChanges,
    [ActionTypes.CANCEL_CHANGES_SUCCESS]: cancelChangesSuccess,
    [ActionTypes.CANCEL_CHANGES_ERROR]: cancelChangesError,
    [ActionTypes.PRESS_CANCEL_CHANGES]: showCancelChangesModal,
    [ActionTypes.DECLINE_CANCEL_CHANGES]: declineCancelChanges
}, initialState);
