import fetch from 'client/common/fetch';
import { serverUrls } from 'common/server-urls';

import { all, call, takeLatest, put } from 'redux-saga/effects';

import {
    loadFileSuccess,
    loadFileFailed,
    downloadJSONSuccess,
    downloadJSONFailed,
    publishSuccess,
    publishFailed,
    startEditSuccess,
    startEditError,
    saveDraftSuccess,
    saveDraftError,
    sendDraftForModerationSuccess,
    sendDraftForModerationError,
    deleteDraftSuccess,
    deleteDraftError,
    cancelChangesSuccess,
    cancelChangesError
} from './actions';

import {
    ActionTypes,
    ILoadFile,
    IDownloadJSON,
    IPublishExam,
    IStartEdit,
    ISaveDraft,
    IDeleteDraft,
    ISendDraftForModeration
} from './types';

import {
    editDataToExam
} from './helpers';

function *loadFile(action: ILoadFile) {
    try {
        const { examSlug, data } = action.payload;

        const uploadUrl = serverUrls.upload.build();
        const fileData = yield call(fetch, uploadUrl, {
            method: 'POST',
            body: data
        });

        const downloadUrl = serverUrls.downloadJSON.build({ examIdentity: examSlug });
        const publishedData = yield call(fetch, downloadUrl);

        const lockUrl = serverUrls.lockExam.build({ examIdentity: examSlug });

        yield call(fetch, lockUrl, { method: 'POST' });

        yield put(loadFileSuccess(fileData, publishedData));
    } catch (err) {
        yield put(loadFileFailed(err.data || err));
    }
}

function *publish(action: IPublishExam) {
    try {
        const { examId, editData, examSlug } = action.payload;
        const draft = editDataToExam(editData, examId);
        const url = serverUrls.publishExam.build();

        yield call(fetch, url, {
            method: 'POST',
            body: JSON.stringify({
                draft,
                examId
            }),
            headers: {
                'Content-Type': 'application/json'
            }
        });

        const unlockUrl = serverUrls.unlockExam.build({ examIdentity: examSlug });

        yield call(fetch, unlockUrl, {
            method: 'POST'
        });

        yield put(publishSuccess());
    } catch (err) {
        yield put(publishFailed(err.data || err));
    }
}

function *downloadJSON(action: IDownloadJSON) {
    try {
        const { examSlug } = action.payload;
        const url = serverUrls.downloadJSON.build({ examIdentity: examSlug });

        const res = yield call(fetch, url);

        yield put(downloadJSONSuccess(res));
    } catch (err) {
        yield put(downloadJSONFailed(err.data || err));
    }
}

function *startEdit(action: IStartEdit) {
    try {
        const { examSlug } = action.payload;

        const getLastVersionUrl = serverUrls.lastVersion.build({ examIdentity: examSlug });
        const lastVersion = yield call(fetch, getLastVersionUrl);

        const downloadUrl = serverUrls.downloadJSON.build({ examIdentity: examSlug });
        const publishedData = yield call(fetch, downloadUrl);

        const lockUrl = serverUrls.lockExam.build({ examIdentity: examSlug });

        yield call(fetch, lockUrl, { method: 'POST' });

        yield put(startEditSuccess(lastVersion, publishedData));
    } catch (err) {
        yield put(startEditError(err.data || err));
    }
}

function *saveDraft(action: ISaveDraft) {
    try {
        const { examSlug, examId, editData } = action.payload;
        const body = editDataToExam(editData, examId);
        const saveDraftUrl = serverUrls.draft.build({ examIdentity: examSlug });

        yield call(fetch, saveDraftUrl, {
            method: 'POST',
            body: JSON.stringify({ draft: body }),
            headers: {
                'Content-Type': 'application/json'
            }
        });

        const unlockUrl = serverUrls.unlockExam.build({ examIdentity: examSlug });

        yield call(fetch, unlockUrl, {
            method: 'POST'
        });

        yield put(saveDraftSuccess());
    } catch (err) {
        yield put(saveDraftError(err.data || err));
    }
}

function *sendDraftForModeration(action: ISendDraftForModeration) {
    try {
        const { examSlug, examId, editData } = action.payload;
        const draft = editDataToExam(editData, examId);
        const sendDraftForModerationUrl = serverUrls.draftModeration.build();

        yield call(fetch, sendDraftForModerationUrl, {
            method: 'POST',
            body: JSON.stringify({
                draft,
                examIdentity: examSlug
            }),
            headers: {
                'Content-Type': 'application/json'
            }
        });

        const unlockUrl = serverUrls.unlockExam.build({ examIdentity: examSlug });

        yield call(fetch, unlockUrl, {
            method: 'POST'
        });

        yield put(sendDraftForModerationSuccess());
    } catch (err) {
        yield put(sendDraftForModerationError(err.data || err));
    }
}

function *deleteDraft(action: IDeleteDraft) {
    try {
        const { examSlug } = action.payload;
        const deleteDraftUrl = serverUrls.draft.build({ examIdentity: examSlug });

        yield call(fetch, deleteDraftUrl, {
            method: 'DELETE',
            body: JSON.stringify({}),
            headers: {
                'Content-Type': 'application/json'
            }
        });

        const unlockUrl = serverUrls.unlockExam.build({ examIdentity: examSlug });

        yield call(fetch, unlockUrl, {
            method: 'POST'
        });

        yield put(deleteDraftSuccess());
    } catch (err) {
        yield put(deleteDraftError(err.data || err));
    }
}

function *cancelChanges(action: IDeleteDraft) {
    try {
        const { examSlug } = action.payload;
        const unlockUrl = serverUrls.unlockExam.build({ examIdentity: examSlug });

        yield call(fetch, unlockUrl, {
            method: 'POST'
        });

        yield put(cancelChangesSuccess());
    } catch (err) {
        yield put(cancelChangesError(err.data || err));
    }
}

export default function *() {
    yield all([
        takeLatest(ActionTypes.LOAD_FILE, loadFile),
        takeLatest(ActionTypes.DOWNLOAD_JSON, downloadJSON),
        takeLatest(ActionTypes.PUBLISH_EXAM, publish),
        takeLatest(ActionTypes.START_EDIT, startEdit),
        takeLatest(ActionTypes.SAVE_DRAFT, saveDraft),
        takeLatest(ActionTypes.SEND_DRAFT_FOR_MODERATION, sendDraftForModeration),
        takeLatest(ActionTypes.DELETE_DRAFT, deleteDraft),
        takeLatest(ActionTypes.CANCEL_CHANGES, cancelChanges)
    ]);
}
