import { handleActions } from 'redux-actions';

import { validateNumber } from 'client/store/common/helpers';

import {
    ActionTypes,
    INullifyAttemptState,
    NullifyAttemptsPayload,
    ITypeAttemptIdsAction,
    IGetAttemptsInfoSuccessAction,
    IGetAttemptsInfoErrorAction,
    ISubmitErrorAction,
    ISubmitSuccessAction
} from './types';

const initialState: INullifyAttemptState = {
    attemptIds: '',
    hasFocusAttemptIds: false,
    isValidAttemptIds: true,
    hasAttemptIds: false,
    isPending: false,
    requestError: null,
    isVisibleConfirmationModal: false,
    attemptsInfo: null,
    isNullifySuccess: false,
    notFoundAttempts: null
};

export function parseAttemptIds(attemptIds: string): string[] {
    return attemptIds
        .replace(/\s+/g, '')
        .split(',')
        .filter(attemptId => {
            return attemptId.length;
        });
}

function typeAttemptIds(
    state: INullifyAttemptState,
    action: ITypeAttemptIdsAction
): INullifyAttemptState {
    const { inputText } = action.payload;
    const attemptIds = parseAttemptIds(inputText);

    const isValidAttemptIds = attemptIds.every(validateNumber);

    return Object.assign({}, state, {
        attemptIds: inputText,
        isValidAttemptIds,
        hasAttemptIds: inputText.length > 0,
        isNullifySuccess: false,
        notFoundAttempts: null
    });
}

function focusAttemptIds(state: INullifyAttemptState): INullifyAttemptState {
    return Object.assign({}, state, { hasFocusAttemptIds: true });
}

function blurAttemptIds(state: INullifyAttemptState): INullifyAttemptState {
    return Object.assign({}, state, { hasFocusAttemptIds: false });
}

function getAttemptsInfo(state: INullifyAttemptState): INullifyAttemptState {
    return Object.assign({}, state, {
        isPending: true,
        requestError: null,
        attemptsInfo: null,
        isNullifySuccess: false,
        notFoundAttempts: null
    });
}

function getAttemptsInfoSuccess(
    state: INullifyAttemptState,
    action: IGetAttemptsInfoSuccessAction
): INullifyAttemptState {
    const { attemptsInfo } = action.payload;

    return Object.assign({}, state, {
        isPending: false,
        requestError: null,
        attemptsInfo

    });
}

function getAttemptsInfoError(
    state: INullifyAttemptState,
    action: IGetAttemptsInfoErrorAction
): INullifyAttemptState {
    return Object.assign({}, state, {
        isPending: false,
        requestError: action.payload
    });
}

function showConfirmation(state: INullifyAttemptState): INullifyAttemptState {
    return Object.assign({}, state, {
        isVisibleConfirmationModal: true
    });
}

function confirm(state: INullifyAttemptState): INullifyAttemptState {
    return Object.assign({}, state, {
        isVisibleConfirmationModal: false,
        isPending: true
    });
}

function decline(state: INullifyAttemptState): INullifyAttemptState {
    return Object.assign({}, state, { isVisibleConfirmationModal: false });
}

function submitSuccess(
    state: INullifyAttemptState,
    action: ISubmitSuccessAction
): INullifyAttemptState {
    return Object.assign({}, state, {
        isPending: false,
        isNullifySuccess: true,
        requestError: null,
        attemptsInfo: null,
        notFoundAttempts: action.payload.notFoundAttempts
    });
}

function submitError(
    state: INullifyAttemptState,
    action: ISubmitErrorAction
): INullifyAttemptState {
    return Object.assign({}, state, {
        isPending: false,
        requestError: action.payload
    });
}

export default handleActions<INullifyAttemptState, NullifyAttemptsPayload>({
    [ActionTypes.TYPE_ATTEMPT_IDS]: typeAttemptIds,
    [ActionTypes.FOCUS_ATTEMPT_IDS]: focusAttemptIds,
    [ActionTypes.BLUR_ATTEMPT_IDS]: blurAttemptIds,
    [ActionTypes.GET_ATTEMPTS_INFO]: getAttemptsInfo,
    [ActionTypes.GET_ATTEMPTS_INFO_SUCCESS]: getAttemptsInfoSuccess,
    [ActionTypes.GET_ATTEMPTS_INFO_ERROR]: getAttemptsInfoError,
    [ActionTypes.PRESS_NULLIFY]: showConfirmation,
    [ActionTypes.CONFIRM]: confirm,
    [ActionTypes.DECLINE]: decline,
    [ActionTypes.SUBMIT_SUCCESS]: submitSuccess,
    [ActionTypes.SUBMIT_ERROR]: submitError
}, initialState);
