import { handleActions } from 'redux-actions';

import { validateNumber } from 'client/store/common/helpers';

import RequestStatus from 'common/types/requestStatus';

import {
    ActionTypes,
    IGetCertsInfoErrorAction,
    IGetCertsInfoSuccessAction,
    INullifyCertificateState,
    ISubmitErrorAction,
    ITypeCertificateIdsAction,
    NullifyCertificatesPayload
} from './types';

const initialState: INullifyCertificateState = {
    certIds: '',
    isValidCertIds: true,
    hasFocusCertIds: false,
    nullifyCertsStatus: null,
    getInfoStatus: null,
    requestError: null,
    isVisibleConfirmationModal: false,
    certificatesInfo: null
};

export function parseCertificateIds(certificateIds: string): string[] {
    return certificateIds
        .replace(/\s+/g, '')
        .split(',')
        .filter(certId => {
            return certId.length;
        });
}

function typeCertificateIds(
    state: INullifyCertificateState,
    action: ITypeCertificateIdsAction
): INullifyCertificateState {
    const { inputText } = action.payload;
    const certIds = parseCertificateIds(inputText);

    const isValidCertIds = certIds.every(validateNumber);

    return Object.assign({}, state, {
        certIds: inputText,
        isValidCertIds,
        nullifyCertsStatus: null,
        getInfoStatus: null
    });
}

function focusCertificateIds(state: INullifyCertificateState): INullifyCertificateState {
    return Object.assign({}, state, {
        hasFocusCertIds: true,
        nullifyCertsStatus: null
    });
}

function blurCertificateIds(state: INullifyCertificateState): INullifyCertificateState {
    return Object.assign({}, state, {
        hasFocusCertIds: false,
        nullifyCertsStatus: null
    });
}

function getCertificatesInfo(state: INullifyCertificateState): INullifyCertificateState {
    return Object.assign({}, state, {
        nullifyCertsStatus: null,
        getInfoStatus: RequestStatus.pending,
        requestError: null,
        certificatesInfo: null
    });
}

function getCertificatesInfoSuccess(
    state: INullifyCertificateState,
    action: IGetCertsInfoSuccessAction
): INullifyCertificateState {
    const { certificatesInfo } = action.payload;

    return Object.assign({}, state, {
        nullifyCertsStatus: null,
        getInfoStatus: RequestStatus.success,
        requestError: null,
        certificatesInfo
    });
}

function getCertificatesInfoError(
    state: INullifyCertificateState,
    action: IGetCertsInfoErrorAction
): INullifyCertificateState {
    return Object.assign({}, state, {
        nullifyCertsStatus: null,
        getInfoStatus: RequestStatus.failed,
        requestError: action.payload
    });
}

function showConfirmation(state: INullifyCertificateState): INullifyCertificateState {
    return Object.assign({}, state, { isVisibleConfirmationModal: true });
}

function confirm(state: INullifyCertificateState): INullifyCertificateState {
    return Object.assign({}, state, {
        isVisibleConfirmationModal: false,
        nullifyCertsStatus: RequestStatus.pending,
        requestError: null
    });
}

function decline(state: INullifyCertificateState): INullifyCertificateState {
    return Object.assign({}, state, { isVisibleConfirmationModal: false });
}

function submitSuccess(state: INullifyCertificateState): INullifyCertificateState {
    return Object.assign({}, state, {
        nullifyCertsStatus: RequestStatus.success,
        requestError: null,
        certificatesInfo: null
    });
}

function submitError(
    state: INullifyCertificateState,
    action: ISubmitErrorAction
): INullifyCertificateState {
    return Object.assign({}, state, {
        nullifyCertsStatus: RequestStatus.failed,
        requestError: action.payload
    });
}

export default handleActions<INullifyCertificateState, NullifyCertificatesPayload>({
    [ActionTypes.TYPE_CERTIFICATE_IDS]: typeCertificateIds,
    [ActionTypes.FOCUS_CERTIFICATE_IDS]: focusCertificateIds,
    [ActionTypes.BLUR_CERTIFICATE_IDS]: blurCertificateIds,
    [ActionTypes.GET_CERTIFICATES_INFO]: getCertificatesInfo,
    [ActionTypes.GET_CERTIFICATES_INFO_SUCCESS]: getCertificatesInfoSuccess,
    [ActionTypes.GET_CERTIFICATES_INFO_ERROR]: getCertificatesInfoError,
    [ActionTypes.PRESS_NULLIFY]: showConfirmation,
    [ActionTypes.CONFIRM]: confirm,
    [ActionTypes.DECLINE]: decline,
    [ActionTypes.SUBMIT_SUCCESS]: submitSuccess,
    [ActionTypes.SUBMIT_ERROR]: submitError
}, initialState);
