import { handleActions } from 'redux-actions';

import { validateLogin } from 'client/store/common/helpers';
import RequestStatus from 'common/types/requestStatus';

import {
    ActionTypes,
    IChooseExamsAction,
    ITypeReasonAction,
    IUnbanErrorAction,
    IUnbanFormState,
    UnbanFormPayload,
    ITypeLoginAction
} from './types';

const initialState: IUnbanFormState = {
    examsForUnban: [],
    userLogin: '',
    reason: '',
    isSuperUnban: false,
    unbanStatus: null,
    isVisibleConfirmationModal: false,
    requestError: null,
    submitErrors: null
};

interface IValidateData {
    userLogin: string,
    reason: string,
    examsForUnban: string[],
    isSuperUnban: boolean
}

// eslint-disable-next-line
function validateUnbanData(validateData: IValidateData) {
    const { userLogin, reason, examsForUnban, isSuperUnban } = validateData;
    const validationErrors = [];
    const parsedLogin = userLogin.trim();
    const isLoginValid = parsedLogin.length && validateLogin(parsedLogin);

    if (!isLoginValid) {
        validationErrors.push('Введите корректный логин');
    }

    if (!reason.length) {
        validationErrors.push('Введите причину');
    }

    if (!isSuperUnban && !examsForUnban.length) {
        validationErrors.push('Выберите экзамены');
    }

    return validationErrors;
}

function chooseExams(state: IUnbanFormState, action: IChooseExamsAction): IUnbanFormState {
    return Object.assign({}, state, {
        examsForUnban: action.payload.exams,
        requestError: null,
        submitErrors: null,
        unbanStatus: null
    });
}

function typeLogin(state: IUnbanFormState, action: ITypeLoginAction): IUnbanFormState {
    const { inputText } = action.payload;

    return Object.assign({}, state, {
        userLogin: inputText,
        requestError: null,
        submitErrors: null,
        unbanStatus: null
    });
}

function typeReason(state: IUnbanFormState, action: ITypeReasonAction) : IUnbanFormState {
    const { inputText } = action.payload;

    return Object.assign({}, state, {
        reason: inputText,
        requestError: null,
        submitErrors: null,
        unbanStatus: null
    });
}

function toggleSuperUnban(state: IUnbanFormState): IUnbanFormState {
    const { isSuperUnban } = state;

    return Object.assign({}, state, {
        isSuperUnban: !isSuperUnban,
        requestError: null,
        submitErrors: null,
        unbanStatus: null
    });
}

function pressUnban(state: IUnbanFormState): IUnbanFormState {
    const { userLogin, reason, examsForUnban, isSuperUnban } = state;
    const validationErrors = validateUnbanData({
        userLogin,
        reason,
        examsForUnban,
        isSuperUnban
    });
    const isCorrectData = validationErrors.length === 0;

    return Object.assign({}, state, {
        isVisibleConfirmationModal: isCorrectData,
        submitErrors: isCorrectData ? null : validationErrors,
        unbanStatus: null
    });
}

function decline(state: IUnbanFormState): IUnbanFormState {
    return Object.assign({}, state, { isVisibleConfirmationModal: false });
}

function unban(state: IUnbanFormState): IUnbanFormState {
    return Object.assign({}, state, {
        unbanStatus: RequestStatus.pending,
        isVisibleConfirmationModal: false,
        requestError: null,
        submitErrors: null
    });
}

function unbanSuccess(state: IUnbanFormState): IUnbanFormState {
    return Object.assign({}, state, {
        unbanStatus: RequestStatus.success,
        examsForUnban: [],
        isSuperUnban: false
    });
}

function unbanError(state: IUnbanFormState, action: IUnbanErrorAction): IUnbanFormState {
    return Object.assign({}, state, {
        unbanStatus: RequestStatus.failed,
        requestError: action.payload
    });
}

export default handleActions<IUnbanFormState, UnbanFormPayload>({
    [ActionTypes.CHOOSE_EXAMS]: chooseExams,
    [ActionTypes.TYPE_LOGIN]: typeLogin,
    [ActionTypes.TYPE_REASON]: typeReason,
    [ActionTypes.TOGGLE_SUPER_UNBAN]: toggleSuperUnban,
    [ActionTypes.PRESS_UNBAN]: pressUnban,
    [ActionTypes.DECLINE]: decline,
    [ActionTypes.UNBAN]: unban,
    [ActionTypes.UNBAN_SUCCESS]: unbanSuccess,
    [ActionTypes.UNBAN_ERROR]: unbanError
}, initialState);
