import { handleActions } from 'redux-actions';

import { validateEmail, validateNumber } from '../common/helpers';
import { validateAppealData } from '../../common/validate';
import RequestStatus from '../../../common/types/requestStatus';

import {
    ActionTypes,
    IDecideAction,
    IGetVideosErrorAction,
    IGetVideosSuccessAction,
    ISubmitErrorAction,
    ITypeAttemptIdAction,
    ITypeEmailAction,
    IVideosState,
    VideosPayload
} from './types';

const initialState: IVideosState = {
    attemptId: '',
    hasFocusAttemptId: false,
    isValidAttemptId: true,
    hasAttemptId: true,
    videos: [],
    verdict: null,
    email: '',
    hasFocusEmail: false,
    isValidEmail: true,
    hasEmail: true,
    isSubmitError: false,
    requestStatus: null,
    isVisibleConfirmationModal: false,
    isAppealSuccess: false,
    requestError: null,
    appealDataFromDB: null,
    pdfLink: '',
    proctoringResponsesInfo: null
};

function typeAttemptId(state: IVideosState, action: ITypeAttemptIdAction): IVideosState {
    const { inputText } = action.payload;
    const isValidAttemptId = validateNumber(inputText);

    return Object.assign({}, state, {
        attemptId: isValidAttemptId ? inputText : state.attemptId,
        isValidAttemptId,
        hasAttemptId: inputText.length > 0,
        isSubmitError: false
    });
}

function focusAttemptId(state: IVideosState): IVideosState {
    return Object.assign({}, state, { hasFocusAttemptId: true });
}

function blurAttemptId(state: IVideosState): IVideosState {
    return Object.assign({}, state, { hasFocusAttemptId: false });
}

function getVideos(state: IVideosState): IVideosState {
    return Object.assign({}, state, {
        email: '',
        requestStatus: RequestStatus.pending,
        isAppealSuccess: false,
        requestError: null,
        appealDataFromDB: null,
        pdfLink: '',
        proctoringResponsesInfo: null
    });
}

function getVideosSuccess(state: IVideosState, action: IGetVideosSuccessAction): IVideosState {
    const { videos, appealData, pdfLink, additionalInfo } = action.payload;

    return Object.assign({}, state, {
        requestStatus: RequestStatus.success,
        requestError: null,
        verdict: null,
        videos,
        appealDataFromDB: appealData,
        pdfLink,
        proctoringResponsesInfo: additionalInfo
    });
}

function getVideosError(state: IVideosState, action: IGetVideosErrorAction): IVideosState {
    return Object.assign({}, state, {
        requestStatus: RequestStatus.failed,
        requestError: action.payload
    });
}

function decide(state: IVideosState, action: IDecideAction): IVideosState {
    const { verdict } = action.payload;

    return Object.assign({}, state, { verdict });
}

function typeEmail(state: IVideosState, action: ITypeEmailAction): IVideosState {
    const { inputText } = action.payload;

    return Object.assign({}, state, {
        email: inputText,
        isValidEmail: validateEmail(inputText),
        hasEmail: inputText.length > 0,
        isSubmitError: false
    });
}

function focusEmail(state: IVideosState): IVideosState {
    return Object.assign({}, state, { hasFocusEmail: true });
}

function blurEmail(state: IVideosState): IVideosState {
    return Object.assign({}, state, { hasFocusEmail: false });
}

function submit(state: IVideosState): IVideosState {
    const { attemptId, email, isValidAttemptId, isValidEmail, verdict } = state;
    const hasValidValues = validateAppealData({
        attemptId,
        email,
        isValidAttemptId,
        isValidEmail,
        verdict
    });

    return Object.assign({}, state, {
        isVisibleConfirmationModal: hasValidValues,
        isSubmitError: !hasValidValues
    });
}

function confirm(state: IVideosState): IVideosState {
    return Object.assign({}, state, {
        isVisibleConfirmationModal: false,
        requestStatus: RequestStatus.pending
    });
}

function decline(state: IVideosState): IVideosState {
    return Object.assign({}, state, { isVisibleConfirmationModal: false });
}

function submitSuccess(state: IVideosState): IVideosState {
    return Object.assign({}, state, {
        requestStatus: RequestStatus.success,
        isAppealSuccess: true,
        requestError: null
    });
}

function submitError(state: IVideosState, action: ISubmitErrorAction): IVideosState {
    return Object.assign({}, state, {
        requestStatus: RequestStatus.failed,
        requestError: action.payload
    });
}

export default handleActions<IVideosState, VideosPayload>({
    [ActionTypes.TYPE_ATTEMPT_ID]: typeAttemptId,
    [ActionTypes.FOCUS_ATTEMPT_ID]: focusAttemptId,
    [ActionTypes.BLUR_ATTEMPT_ID]: blurAttemptId,
    [ActionTypes.GET_VIDEOS]: getVideos,
    [ActionTypes.GET_VIDEOS_SUCCESS]: getVideosSuccess,
    [ActionTypes.GET_VIDEOS_ERROR]: getVideosError,
    [ActionTypes.DECIDE]: decide,
    [ActionTypes.TYPE_EMAIL]: typeEmail,
    [ActionTypes.FOCUS_EMAIL]: focusEmail,
    [ActionTypes.BLUR_EMAIL]: blurEmail,
    [ActionTypes.PRESS_SEND]: submit,
    [ActionTypes.CONFIRM]: confirm,
    [ActionTypes.DECLINE]: decline,
    [ActionTypes.SUBMIT_SUCCESS]: submitSuccess,
    [ActionTypes.SUBMIT_ERROR]: submitError
}, initialState);
