import Cookies from 'js-cookie';

import { APIError } from './error';

interface IFetchOptions extends Partial<RequestInit> {
    excludeDefaultHeaders?: boolean;
}

function isSafeMethod(methodName: string = 'GET') {
    return ['GET', 'HEAD', 'OPTIONS'].indexOf(methodName) > -1;
}

const DEFAULT_HEADERS: HeadersInit = {
    'Content-Type': 'application/json'
};

export default async function <T> (path: string, options: IFetchOptions = {}): Promise<T> {
    const { headers = {}, excludeDefaultHeaders = false, ...restOptions } = options;
    const securityHeaders: HeadersInit = {};

    if (!isSafeMethod(options.method)) {
        securityHeaders['X-CSRF-Token'] = Cookies.get('csrftoken') || '';
    }

    const defaultHeaders = excludeDefaultHeaders ? {} : DEFAULT_HEADERS;

    const response = await fetch(path, {
        headers: {
            ...defaultHeaders,
            ...securityHeaders,
            ...headers,
            Accept: 'application/json'
        },
        credentials: 'same-origin',
        ...restOptions
    });

    if (!response.ok) {
        await handleErrorResponse(response);
    }

    return await handleSuccessResponse(response);
}

async function handleSuccessResponse(response: Response) {
    if (response.status === 204) {
        return;
    }

    return await response.json();
}

async function handleErrorResponse(response: Response) {
    const { url, status, statusText } = response;

    let data;
    let code;

    try {
        data = await response.json();
    } finally {
        code = (data && data.code) || `${status}_error`;
    }

    throw new APIError(code, { url, status, statusText, data });
}
