import { Request, Response, NextFunction as Next } from 'express';
import config from 'yandex-cfg';

import Dictionary from 'common/types/dictionary';
import IExam from 'common/types/exam';
import IData from 'common/types/data';

import { ApplicationState } from 'client/store';

const { exams } = config;

export default class BaseController {
    protected readonly req: Request;
    protected readonly res: Response;
    protected readonly next?: Next;

    protected constructor(req: Request, res: Response, next?: Next) {
        this.req = req;
        this.res = res;
        this.next = next;
    }

    public static getAction(actionName: string) {
        return async function (req: Request, res: Response, next: Next): Promise<void> {
            const controller = new this(req, res, next);

            try {
                await controller[actionName]();
            } catch (error) {
                console.error(error);
                controller.renderError();
            }
        }.bind(this);
    }

    public render(data: IData, state?: Partial<ApplicationState>): void {
        this.res.renderBundle('desktop', this.getData(this.req, data), this.getState(state));
    }

    private getState(state?: Partial<ApplicationState>): Partial<ApplicationState> {
        return { ...state };
    }

    private getData(req: Request, data: IData) {
        const resHeaders = this.res.getHeaders();

        const faviconLang = 'ru';
        const {
            blackbox,
            geolocation,
            headers: reqHeaders,
            ip,
            langdetect,
            nonce,
            secretkey,
            uatraits,
            base
        } = req;

        return {
            blackbox,
            faviconLang,
            geolocation,
            ip,
            langdetect,
            nonce,
            reqHeaders,
            resHeaders,
            secretkey,
            uatraits,
            base,
            examSlugs: exams.reduce((acc: Dictionary<string>, exam: IExam) => {
                const { slug } = exam;

                acc[slug] = slug;

                return acc;
            }, {}),
            ...data
        };
    }

    public renderError() {
        this.res.sendStatus(404);
    }
}
