import cfg from 'yandex-cfg';
import cookieParserMiddleware from 'cookie-parser';
import { ErrorRequestHandler, Request, RequestHandler } from 'express';
import expressBlackboxMiddleware from 'express-blackbox';
import expressHttpGeobaseMiddleware from 'express-http-geobase';
import expressHttpLangdetectMiddleware from 'express-http-langdetect';
import expressHttpUatraitsMiddleware from 'express-http-uatraits';
import expressRenewBbSessionMiddleware from 'express-renew-bb-session';
import expressSecretkey from 'express-secretkey';
import expressTldMiddleware from 'express-tld';
import expressTvmMiddleware from '@yandex-int/express-tvm';
import expressXForwardedForFix from 'express-x-forwarded-for-fix';
import expressYandexCsp from 'express-yandex-csp';
import expressYandexuid from 'express-yandexuid';
import helmetMiddleware from 'helmet';

import mockBlackboxMiddleware from '../mock/blackbox';

import { AccessHandler } from '../types/access';

import expressBase from './express-base';
import finalErrorHandlerMiddleware from './final-error-handler';
import loggerMiddleware from './logger';
import renderMiddleware from './render';
import accessMiddleware from './access';

// Mocks blackbox for local development
export const mockBlackbox: RequestHandler = mockBlackboxMiddleware();

// Gets yandex user data into req.blackbox
export const blackbox: RequestHandler = expressBlackboxMiddleware({
    ...cfg.blackbox,
    getServiceTicket: (req: Request) => {
        if (req.tvm && req.tvm.tickets && req.tvm.tickets.blackbox) {
            return req.tvm.tickets.blackbox.ticket;
        }
    }
});

// Parses cookies into req.cookies
export const cookieParser: RequestHandler = cookieParserMiddleware();

// Logs error and sends 500
export const finalErrorHandler: ErrorRequestHandler = finalErrorHandlerMiddleware;

// Sets secure headers:
// X-dns-prefetch-control: off
// X-frame-options: SAMEORIGIN
// Strict-transport-security: max-age=15552000; includeSubDomains
// X-download-options: noopen
// X-content-type-options: nosniff
// X-xss-protection: 1; mode=block
export const helmet = helmetMiddleware(cfg.helmet);

// Detects user geolocation into req.geolocation
export const httpGeobase: RequestHandler = expressHttpGeobaseMiddleware(cfg.httpGeobase);

// Detects user language into req.langdetect
export const httpLangdetect: RequestHandler = expressHttpLangdetectMiddleware(cfg.httpLangdetect);

// Parses user agent header into req.uatraits
export const httpUatraits: RequestHandler = expressHttpUatraitsMiddleware(cfg.httpUatraits);

// Adds logger instance associated with current request into req.logger
export const logger: RequestHandler = loggerMiddleware();

// Check user access to the page
export const access: AccessHandler = accessMiddleware;

// Adds render function into response object
export const render: RequestHandler | RequestHandler[] = renderMiddleware;

// Renews session cookie if necessary (authorization status is NEED_RESET)
export const renewBbSession: RequestHandler = expressRenewBbSessionMiddleware();

// Generate secret token into req.secretkey and validate it from headers
export const secretkey: RequestHandler = expressSecretkey(cfg.secretkey);

// Parses top level domain into req.tld
export const tld: RequestHandler = expressTldMiddleware();

// Gets TVM service tickets into req.tvmTickets
export const tvm = expressTvmMiddleware(cfg.tvm);

// Filters inccorect IPs from x-forwarded-for header
export const xForwardedForFix = expressXForwardedForFix(cfg.xForwardedForFix);

// Sets Content Security Policy headers and adds nonce into req.nonce
export const yandexCsp = expressYandexCsp(cfg.csp);

// Sets yandexuid cookie
export const yandexuid = expressYandexuid(cfg.yandexuid);

// Sets common used data
export const base = expressBase();
