import cfg from 'yandex-cfg';
import React from 'react';
import Helmet, { HelmetData } from 'react-helmet';
import htmlescape from 'htmlescape';
import { renderToString, renderToStaticMarkup } from 'react-dom/server';
import { StaticRouter } from 'react-router-dom';
import { StaticRouterContext } from 'react-router';

import vendorsMeta from 'webpack-configs/vendors-meta';
import { ApplicationState } from 'client/store';
import { BundleData } from 'client/bundles/types';

function getBundle(bundleName: string, lang: string) {
    const module = `../../../ssr.bundles.${lang}`;

    if (cfg.render && cfg.render.isHot) {
        delete require.cache[require.resolve(module)];
    }

    return require(module).bundles[bundleName]; // eslint-disable-line global-require
}

interface RenderBundleArguments {
    bundleName: string,
    data: BundleData,
    state: ApplicationState,
    lang: string,
    location: string
}

/* eslint-disable-next-line max-statements, complexity */
export default ({ bundleName, data, state, lang, location }: RenderBundleArguments) => {
    const context: StaticRouterContext = {};
    const Bundle = getBundle(bundleName, lang);

    if (!Bundle) {
        throw new Error(`Bundle ${bundleName} not found`);
    }

    const bundleHtml = renderToString(
        <StaticRouter context={context} location={location}>
            <Bundle data={data} state={state} />
        </StaticRouter>
    );

    if (context.url) {
        return { redirectUrl: context.url };
    }

    const helmet = Helmet.rewind();

    return {
        html: getPageHtml({
            helmet,
            bundleName,
            bundleHtml,
            data,
            lang,
            state
        })
    };
};

interface pageHtmlParams {
    helmet: HelmetData,
    bundleName: string,
    bundleHtml: string,
    data: BundleData,
    state: ApplicationState,
    lang: string
}

function getPageHtml(params: pageHtmlParams) {
    const { helmet, bundleName, bundleHtml, data, lang, state = {} } = params;
    const { baseUrl, version, frozenPath } = cfg.static;
    const bundleFilePath = `${baseUrl}${version}/${bundleName}.bundle`;
    const vendorsFilePath = `${baseUrl}${frozenPath}/${vendorsMeta.name}`;

    const html = renderToStaticMarkup(
        <html>
            <head>
                {helmet.title.toComponent()}
                {helmet.meta.toComponent()}
                {helmet.link.toComponent()}
                {helmet.script.toComponent()}
                <link rel="stylesheet" href={`${bundleFilePath}.css`} />
                {vendorsMeta.hasCss && <link rel="stylesheet" href={`${vendorsFilePath}.css`} />}
            </head>
            <body>
                <div id="mount" dangerouslySetInnerHTML={{ __html: bundleHtml }} />
                {vendorsMeta.hasJs && <script nonce={data.nonce} src={`${vendorsFilePath}.js`} />}
                <script nonce={data.nonce} src={`${bundleFilePath}.${lang}.js`} />
                <script
                    nonce={data.nonce}
                    dangerouslySetInnerHTML={{
                        __html: `Client.default(${htmlescape(data)}, ${htmlescape(state)})`
                    }}
                    />
            </body>
        </html>
    );

    return `<!doctype html>${html}`;
}
