import { Agent } from 'http';

import { Request } from 'express';
import _ from 'lodash';
import config from 'yandex-cfg';

import IData from 'common/types/data';

import apiHost from '../lib/apiHost';

import {
    IAttributes,
    IRequestOptionsHeaders,
    IRequestOptions,
    IRequestParams
} from '../types/request';

import request from '../request';

const keepAliveAgent = new Agent({ keepAlive: true, maxSockets: 1024 });

export default class BaseModel {
    protected _req: Request;
    protected _attrs: IAttributes;
    protected _ip: string;
    protected _userhost?: string;
    protected _cookie?: string | string[];
    protected _data: IData;

    constructor(req: Request, attrs: IAttributes, data?: IData) {
        this._req = req;
        this._attrs = attrs;
        this._ip = config.defaultIp || req.ip;
        this._userhost = req.headers.host;
        this._cookie = req.headers.cookie;

        if (data) {
            this._data = data;
        }
    }

    protected _removeEmptyProperties<T extends object>(obj: T): T {
        for (const key in obj) {
            if (obj[key] === undefined) {
                delete obj[key];
            }
        }

        return obj;
    }

    /**
     * Получение данных модели
     *
     * @returns {IData}
     */
    public toJSON(): IData {
        return this._data;
    }

    /**
     * Получение значения свойства
     *
     * @param {String} name имя
     * @returns {any}
     */
    public get(name: string): any {
        return _.get(this._data, name);
    }

    protected _prepareRequestOptions(options: IRequestOptions): IRequestParams {
        options = _.extend(options, {
            agent: keepAliveAgent,
            retries: 0
        });
        const headers: IRequestOptionsHeaders = _.extend({
            userhost: this._userhost,
            userip: this._ip,
            cookie: this._cookie,
            'accept-encoding': 'gzip',
            'content-type': 'application/json'
        }, options.headers);

        options.headers = this._removeEmptyProperties(headers);

        if (options.body) {
            options.body = JSON.stringify(options.body);
        }

        options.query = options.query && this._removeEmptyProperties(options.query);
        const url = apiHost + options.url;

        delete options.url;

        return {
            options,
            url
        };
    }

    protected async _request<R extends object>(options: IRequestOptions) {
        const params = this._prepareRequestOptions(options);

        params.options.json = true;

        return await this._fetchData<R>(params.url, params.options);
    }

    protected async _fetchData<R extends object>(url: string, options: IRequestOptions) {
        return await request<R>(url, options);
    }

    protected parseData(): any {
        return this._data;
    }
}
